package ru.yandex.crm.apphost.kotlin.tools.entityammogenerator

import com.google.protobuf.GeneratedMessageV3
import com.google.protobuf.Timestamp
import com.google.protobuf.util.JsonFormat
import picocli.CommandLine.Command
import picocli.CommandLine.Option
import ru.yandex.crm.apphost.proto.entitystorage.Entitystorage.CreateEntityCommand
import ru.yandex.crm.apphost.proto.entitystorage.Entitystorage.ListEntitesCommand
import ru.yandex.crm.apphost.proto.userselector.UserSelector
import ru.yandex.crm.proto.gallifrey.entitystorage.Entitystorage
import ru.yandex.crm.proto.gallifrey.entitystorage.Entitystorage.AttributeSchema
import ru.yandex.crm.proto.gallifrey.entitystorage.Entitystorage.AttributeSchema.AttributeType
import ru.yandex.crm.proto.gallifrey.entitystorage.Entitystorage.AttributeValue
import java.io.File
import java.lang.Math.random
import java.time.Instant
import java.util.Base64
import java.util.UUID
import java.util.concurrent.Callable
import kotlin.random.Random

@Command(name = "entity-ammo-generator", mixinStandardHelpOptions = true, version = ["1.1"])
class CreateEntityAmmo : Callable<Int> {

    @Option(
        names = ["-c", "--count"],
        description = ["Ammo count (default: \${DEFAULT-VALUE})"],
        required = true,
        defaultValue = "10"
    )
    var ammoCount: Int = 10

    @Option(names = ["-f", "--file"], description = ["output file"], required = true)
    lateinit var file: File

    @Option(names = ["--schemas"], description = ["File containing meta guids"])
    lateinit var schemasFile: File

    @Option(
        names = ["-t", "--type"],
        description = ["type of ammo (Valid values: \${COMPLETION-CANDIDATES})"],
        defaultValue = "get"
    )
    var ammoType: AmmoType = AmmoType.GET

    @Option(names = ["-sv", "--string-values"], split = ",", description = ["String values to use"])
    var stringValues: List<String> = listOf()

    @Option(names = ["-sp", "--string-probability"], description = ["Probability of using defined string values"])
    var stringProbability: Double = 0.0

    @Option(names = ["-j", "--json"], description = ["Print jsons instead of encoded strings"])
    var printJsons: Boolean = false

    @Option(names = ["--filter-type"], description = ["Filter type (default: \${DEFAULT-VALUE})"])
    var filterType: AttributeType = AttributeType.STRING

    override fun call(): Int {
        file.writeText("")
        val user = UserSelector.User.newBuilder()
            .setUserId(1)
            .setAuthenticationType(UserSelector.User.AuthenticationType.YaStaffUser)
            .build()

        val organization = UserSelector.Organization.newBuilder()
            .setOrganizationId(1)
            .setPoolId(1)
            .build()

        val encodedUser = getPrintedMessage(user)
        val encodedOrganization = getPrintedMessage(organization)

        if (ammoType == AmmoType.CREATE_SCHEMA_FILE) {
            createEntitySchemas()
        } else {
            val schemaJsons = schemasFile.readLines().map {
                val builder = Entitystorage.EntitySchema.newBuilder()
                JsonFormat.parser().merge(it, builder)
                builder.build()
            }
            for (i in 0 until ammoCount) {
                var jsonRequest: String
                var commandName: String
                var url: String
                var encodedCommand: String

                val entitySchema = schemaJsons[i % schemaJsons.size]

                when (ammoType) {
                    AmmoType.CREATE -> {
                        val createEntityCommand = generateCreateEntityCommand(entitySchema)
                        commandName = "create_entity_command"
                        url = "/entity/create"
                        encodedCommand = getPrintedMessage(createEntityCommand)
                    }
                    AmmoType.GET -> {
                        val getEntityCommand = generateGetEntityByIdCommand(entitySchema)

                        commandName = "get_entity_by_id_command"
                        url = "/entity/get"
                        encodedCommand = getPrintedMessage(getEntityCommand)
                    }
                    AmmoType.LIST -> {
                        val listEntitiesCommand = generateListEntitesCommand(entitySchema)

                        commandName = "list_entities_command"
                        url = "/entity/list"
                        encodedCommand = getPrintedMessage(listEntitiesCommand)
                    }
                    else -> error("Unexpected ammo type")
                }
                jsonRequest = getJsonRequest(commandName, url, encodedCommand, encodedUser, encodedOrganization)
                file.appendText(jsonRequest)
                file.appendText("\n")
            }
        }
        return 0
    }

    private fun generateListEntitesCommand(entitySchema: Entitystorage.EntitySchema?): ListEntitesCommand {
        val builder = ListEntitesCommand
            .newBuilder()
            .setSchema(entitySchema)
        if (filterType == AttributeType.STRING) {
            builder.addFilters(
                Entitystorage.AttributeFilter
                    .newBuilder()
                    .setSchema(entitySchema?.attributesList?.first { it.name == "attribute0" })
                    .setEquality(Entitystorage.EqualityFilter.newBuilder().setValue(
                        AttributeValue
                            .newBuilder()
                            .setStringValue(generateRandomValue(null, AttributeType.STRING) as String)
                            .build())
                    )
                    .build()
            )
        } else if (filterType == AttributeType.INT) {
            builder.addFilters(
                Entitystorage.AttributeFilter
                    .newBuilder()
                    .setSchema(entitySchema?.attributesList?.first { it.name == "attribute1" })
                    .setEquality(Entitystorage.EqualityFilter.newBuilder().setValue(
                        AttributeValue
                            .newBuilder()
                            .setIntValue(generateRandomValue(null, AttributeType.INT) as Int)
                            .build())
                    )
                    .build()
            )
        }

        return builder.build()
    }

    private fun generateGetEntityByIdCommand(entitySchema: Entitystorage.EntitySchema?): ListEntitesCommand = ListEntitesCommand
            .newBuilder()
            .setSchema(entitySchema)
            .addFilters(
            Entitystorage.AttributeFilter
                .newBuilder()
                .setSchema(AttributeSchema.newBuilder()
                    .setName("id")
                    .setType(AttributeType.UUID)
                    .build())
                .setEquality(Entitystorage.EqualityFilter.newBuilder().setValue(
                    AttributeValue
                        .newBuilder()
                        .setUuidValue(UUID.randomUUID().toString())
                        .build())
                )
                .build()
        ).build()

    private fun generateCreateEntityCommand(entitySchema: Entitystorage.EntitySchema): CreateEntityCommand {
        val entityBuilder = Entitystorage.EntityRecord
            .newBuilder()
            .setMetaId(entitySchema.metaId)
            .setId(UUID.randomUUID().toString())

        entitySchema.attributesList.forEach { attr ->
            val attribute = Entitystorage.Attribute
                .newBuilder()
                .setSchema(attr)

            generateRandomValue(attribute, attr.type)
            attribute.build()

            entityBuilder.addAttributes(attribute)
        }

        val entity = entityBuilder.build()

        return CreateEntityCommand
            .newBuilder()
            .setEntity(entity)
            .setSchema(entitySchema)
            .build()
    }

    private fun createEntitySchemas() {
        for (i in 0 until ammoCount) {
            printJsons = true
            val builder = Entitystorage.EntitySchema
                .newBuilder()
                .setMetaId(UUID.randomUUID().toString())
                .setOrganizationId(1)

            (0..9).forEach { attrNum ->
                val attributeType = getAttributeTypeByNum(attrNum)
                val attribute =
                    Entitystorage.AttributeSchema
                        .newBuilder()
                        .setName("attribute$attrNum")
                        .setType(attributeType)
                        .build()

                builder.addAttributes(attribute)
            }

            val schema = builder.build()

            val schemaJson = getPrintedMessage(schema)
            file.appendText(schemaJson)
            file.appendText("\n")
        }
    }

    private val charPool: List<Char> = ('a'..'z') + ('A'..'Z') + ('0'..'9')

    private fun generateRandomValue(builder: Entitystorage.Attribute.Builder?, attributeType: AttributeType): Any =
        when (attributeType) {
            AttributeType.STRING -> {
                val value = if (stringValues.isNotEmpty() && Random.nextDouble() < stringProbability) {
                    stringValues.random()
                } else {
                    (1..100)
                        .map { Random.nextInt(0, charPool.size) }
                        .map(charPool::get)
                        .joinToString("")
                }
                builder?.value = AttributeValue.newBuilder().setStringValue(value).build()
                value
            }
            AttributeType.INT -> {
                val value = Random.nextInt(1, 10000)
                builder?.value = AttributeValue.newBuilder().setIntValue(value).build()
                value
            }

            AttributeType.TIMESTAMP -> {
                val value = Instant.now()
                    .minusSeconds(Random.nextInt(1, 10000).toLong())
                val timestampValue = Timestamp
                    .newBuilder().apply {
                        seconds = value.epochSecond
                        nanos = value.nano
                    }.build()
                builder?.value = AttributeValue.newBuilder().setTimestampValue(timestampValue).build()
                timestampValue
            }
            AttributeType.BOOL -> {
                val value = random() > 0.5
                builder?.value = AttributeValue.newBuilder().setBoolValue(value).build()
                value
            }
            AttributeType.UUID -> {
                val value = UUID.randomUUID().toString()
                builder?.value = AttributeValue.newBuilder().setUuidValue(value).build()
                value
            }
            else -> error("Invalid attribute type")
        }

    private fun getPrintedMessage(message: GeneratedMessageV3): String =
        if (printJsons) {
            JsonFormat.printer().print(message).replace("\n", "")
        } else {
            Base64.getEncoder().encodeToString(message.toByteArray())
        }

    private fun getAttributeTypeByNum(attrNum: Int): AttributeType =
        when (attrNum % 5) {
            0 -> AttributeType.STRING
            1 -> AttributeType.INT
            2 -> AttributeType.TIMESTAMP
            3 -> AttributeType.BOOL
            4 -> AttributeType.UUID
            else -> AttributeType.UNRECOGNIZED
        }

    private fun getJsonRequest(
        commandName: String,
        url: String,
        encodedCommand: String,
        encodedUser: String,
        encodedOrganization: String
    ): String =
        "{\"answers\":[{\"source\":\"HTTP_PARSER\",\"type\":\"$commandName\",\"__content_type\"" +
            ":\"protobuf\",\"binary\":\"$encodedCommand\"},{\"source\":\"USER_SELECTOR\",\"type\":\"authenticated_user\",\"__content_type\"" +
            ":\"protobuf\",\"binary\":\"$encodedUser\"},{\"source\":\"USER_SELECTOR\",\"type\":\"user_organization\",\"__content_type\"" +
            ":\"protobuf\",\"binary\":\"$encodedOrganization\"}],\"path\":\"$url\"}"
}

enum class AmmoType {
    CREATE,
    GET,
    LIST,
    CREATE_SCHEMA_FILE
}
