package ru.yandex.crm.apphost.kotlin.tools.timelineentitygenerator

import com.google.protobuf.Message
import com.typesafe.config.ConfigBeanFactory
import com.typesafe.config.ConfigFactory
import io.ktor.client.*
import io.ktor.client.call.*
import io.ktor.client.engine.cio.*
import io.ktor.client.request.*
import io.ktor.client.statement.*
import io.ktor.http.*
import kotlinx.coroutines.coroutineScope
import kotlinx.coroutines.joinAll
import kotlinx.coroutines.launch
import ru.yandex.crm.proto.gallifrey.entitymanager.Entitymanager.EntityRecord
import ru.yandex.crm.proto.gallifrey.entitymanager.Entitymanager.EntityRecordsResponse
import java.util.*

class CreateTimelineEntities {
    private val config = ConfigBeanFactory.create(ConfigFactory.load().getConfig("entityGenerator"), EntityGeneratorConfig::class.java)

    private val httpClient = HttpClient(CIO)

    private val accountGenerator = AccountGenerator()
    private val ticketGenerator = TicketGenerator()
    private val floydMessageGenerator = FloydMessageGenerator()

    suspend fun createEntities() = coroutineScope {
        val departmentId = UUID.randomUUID()

        (1..config.accountCount)
            .map {
                launch {
                    createTickets(departmentId)
                }
            }.joinAll()

        httpClient.close()
    }

    private suspend fun createTickets(departmentId: UUID) {
        val accountEntity = accountGenerator.generate(departmentId)
        val accountRecord = sendRequest(accountEntity, "Account")
        println("Create ${accountRecord.id} account")
        // send http request

        for (ticketNumber in 1..config.ticketsPerAccountCount) {
            val ticketEntity = ticketGenerator.generate(accountRecord.id, departmentId)
            val ticketRecord = sendRequest(ticketEntity, "Ticket")
            println("Create $ticketNumber ticket for ${accountRecord.id} account")
            // send http request

            for (messageNumber in 1..config.messagesPerTicketCount) {
                val messageEntity = floydMessageGenerator.generate(ticketRecord.id, departmentId)
                val messageRecord = sendRequest(messageEntity, "FloydMessage")
                println("Create $messageNumber (${messageRecord.id}) message for ${ticketRecord.id} ticket")
            }
        }
    }

    private suspend fun sendRequest(entity: Message, meta: String): EntityRecord {
        val httpResponse: HttpResponse =
            httpClient.post("${config.serviceUrl}organization/1/entity/${meta}/1") {
                contentType(ContentType.parse("application/x-protobuf"))
                body = entity.toByteArray()
            }

        val responseBytes: ByteArray = httpResponse.receive()
        val entityRecords = EntityRecordsResponse.parseFrom(responseBytes)

        return entityRecords.entityRecordsList.first()
    }
}
