import React, { FC, useCallback, useMemo, useState } from 'react';
import { Tree } from 'components/Tree';
import { useMutation, useQuery } from '@apollo/client';
import { EventDataNode } from 'rc-tree/lib/interface';
import {
    GetOrganizationDepartmentsQuery,
    GetOrganizationDepartmentsQueryResponse,
    GetOrganizationDepartmentsQueryVariables,
} from './GetOrganizationDepartments.query';
import {
    ChangeDepartmentParentMutation,
    ChangeDepartmentParentMutationVariables,
} from './ChangeDepartmentParent.mutation';
import {
    DepartmentsTreeProps,
    FormValues,
    TreeDataNode,
} from './DepartmentsTree.types';
import { departmentsToTreeData } from './DepartmentsTree.utils';
import { DepartmentNode } from './DepartmentNode';
import { context, ContextValue } from './DepartmentsTree.context';
import {
    RenameDepartmentMutation,
    RenameDepartmentMutationVariables,
} from './RenameDepartment.mutation';
import {
    ArchiveDepartmentMutation,
    ArchiveDepartmentMutationVariables,
} from './ArchiveDepartment.mutation';
import {
    CreateChildDepartmentMutation,
    CreateChildDepartmentMutationVariables,
} from './CreateChildDepartment.mutation';

const renderDepartment = (node: TreeDataNode) => {
    return <DepartmentNode node={node} />;
};

export const DepartmentsTree: FC<DepartmentsTreeProps> = props => {
    const { organizationId, onDepartmentSelect } = props;

    const [visibleForm, setVisibleForm] = useState<string>();
    const [formValues, setFormValues] = useState<FormValues>({});

    const { loading, error, data } = useQuery<
        GetOrganizationDepartmentsQueryResponse,
        GetOrganizationDepartmentsQueryVariables
    >(GetOrganizationDepartmentsQuery, {
        variables: {
            organizationId,
        },
    });

    const [changeDepartmentParent] =
        useMutation<ChangeDepartmentParentMutationVariables>(
            ChangeDepartmentParentMutation,
            {
                variables: {
                    organizationId,
                },
                refetchQueries: ['GetOrganizationDepartments'],
            },
        );

    const [renameDepartment] = useMutation<RenameDepartmentMutationVariables>(
        RenameDepartmentMutation,
        {
            variables: {
                organizationId,
            },
            refetchQueries: ['GetOrganizationDepartments'],
            onCompleted: () => {
                setVisibleForm('');
            },
        },
    );

    const [archiveDepartment] = useMutation<ArchiveDepartmentMutationVariables>(
        ArchiveDepartmentMutation,
        {
            variables: {
                organizationId,
            },
            refetchQueries: ['GetOrganizationDepartments'],
        },
    );

    const [createChildDepartment] =
        useMutation<CreateChildDepartmentMutationVariables>(
            CreateChildDepartmentMutation,
            {
                variables: {
                    organizationId,
                },
                refetchQueries: ['GetOrganizationDepartments'],
                onCompleted: () => {
                    setVisibleForm('');
                },
            },
        );

    const handleDrop = useCallback(
        (payload: {
            dragNode: EventDataNode<TreeDataNode>;
            node: EventDataNode<TreeDataNode>;
        }) => {
            const dragNode = payload.dragNode;
            const dropNode = payload.node;

            changeDepartmentParent({
                variables: {
                    departmentId: dragNode.id,
                    parentDepartmentId: dropNode.id,
                },
            });
        },
        [changeDepartmentParent],
    );

    const treeData = useMemo(() => {
        const departments = data?.organization.departments || [];
        return departmentsToTreeData(departments);
    }, [data]);

    const startRenameDepartment = useCallback((departmentId: string) => {
        setVisibleForm('renameDepartment');
        setFormValues({
            departmentId,
        });
    }, []);

    const startAddChildDepartment = useCallback((departmentId: string) => {
        setVisibleForm('addChildDepartment');
        setFormValues({
            departmentId,
        });
    }, []);

    const startChangeDepartmentParent = useCallback((departmentId: string) => {
        setVisibleForm('changeDepartmentParent');
        setFormValues({
            departmentId,
        });
    }, []);

    const contextValue = useMemo<ContextValue>(
        () => ({
            startRenameDepartment,
            startAddChildDepartment,
            startChangeDepartmentParent,
            archiveDepartment: (departmentId: string) => {
                archiveDepartment({
                    variables: {
                        departmentId,
                    },
                });
            },
        }),
        [],
    );

    const handleTreeSelect = (_, payload) => {
        const node = payload.node as TreeDataNode;
        onDepartmentSelect?.(node, payload.selected);
    };

    if (loading) return null;
    if (error) return <>Error! {error}</>;

    return (
        <context.Provider value={contextValue}>
            <Tree<TreeDataNode>
                onSelect={handleTreeSelect}
                draggable
                showLine
                treeData={treeData}
                onDrop={handleDrop}
                titleRender={renderDepartment}
            />

            {visibleForm === 'changeDepartmentParent' && (
                <div>
                    Переместить
                    <input
                        type="text"
                        value={formValues.parentDepartmentId || ''}
                        onChange={event =>
                            setFormValues(prev => ({
                                ...prev,
                                parentDepartmentId: event.target.value,
                            }))
                        }
                    />
                    <button
                        onClick={() => {
                            changeDepartmentParent({
                                variables: {
                                    departmentId: formValues.departmentId,
                                    parentDepartmentId:
                                        formValues.parentDepartmentId,
                                },
                                onCompleted: () => {
                                    setVisibleForm('');
                                },
                            });
                        }}
                    >
                        Сохранить
                    </button>
                </div>
            )}

            {visibleForm === 'addChildDepartment' && (
                <div>
                    Новый департамент
                    <input
                        type="text"
                        placeholder="Название"
                        value={formValues.name || ''}
                        onChange={event =>
                            setFormValues(prev => ({
                                ...prev,
                                name: event.target.value,
                            }))
                        }
                    />
                    <button
                        onClick={() => {
                            createChildDepartment({
                                variables: {
                                    parentDepartmentId: formValues.departmentId,
                                    name: formValues.name,
                                },
                            });
                        }}
                    >
                        Добавить
                    </button>
                </div>
            )}

            {visibleForm === 'renameDepartment' && (
                <div>
                    Переименовать
                    <input
                        type="text"
                        value={formValues.name || ''}
                        onChange={event =>
                            setFormValues(prev => ({
                                ...prev,
                                name: event.target.value,
                            }))
                        }
                    />
                    <button
                        onClick={() =>
                            renameDepartment({
                                variables: {
                                    departmentId: formValues.departmentId,
                                    name: formValues.name,
                                },
                            })
                        }
                    >
                        Сохранить
                    </button>
                </div>
            )}
        </context.Provider>
    );
};
