import React, {
    CSSProperties,
    FC,
    useCallback,
    useMemo,
    useState,
} from 'react';
import { ComponentStory, ComponentMeta } from '@storybook/react';
import cx from 'classnames';
import { DataNode } from 'rc-tree/lib/interface';
import { TreeNode } from '../src/components/TreeNode';
import { Tree } from '../src/components/Tree';
import { treeToData } from '../src/components/DepartmentsTree/DepartmentsTree.utils';

export default {
    /* 👇 The title prop is optional.
     * See https://storybook.js.org/docs/react/configure/overview#configure-story-loading
     * to learn how to generate automatic titles
     */
    title: 'Tree',
    component: Tree,
} as ComponentMeta<typeof Tree>;

export const Primary: ComponentStory<typeof Tree> = () => <Tree />;

const renderDepartment = (node: DataNode) => {
    return <TreeNode title={node.title || 'Hello'} />;
};

export const DepartmentsTree: FC = props => {
    const [tree, setTree] = useState({
        root: ['1', '2'],
        children: {
            2: ['3'],
        },
        nodes: {
            1: {
                id: '1',
                title: '1 department',
                orgId: 1,
            },
            2: {
                id: '2',
                title: '2 depa',
                orgId: 1,
            },
            3: {
                id: '3',
                title: '3 department',
                orgId: 2,
            },
        },
    });

    const treeData = useMemo(() => {
        return treeToData(tree);
    }, [tree]);

    const handleDrop = useCallback(
        payload => {
            const dragNode = payload.dragNode;
            const dropNode = payload.node;
            console.log(payload);

            setTree(prev => {
                const prevParentId = prev.parents[dragNode.id];
                const newParentId = dropNode.id;
                const newChildren = {
                    ...prev.children,
                    [prevParentId]: (prev.children[prevParentId] || []).filter(
                        id => id !== dragNode.id,
                    ),
                    [newParentId]: [
                        ...(prev.children[newParentId] || []),
                        dragNode.id,
                    ],
                };
                const newParents = {
                    ...prev.parents,
                    [dragNode.id]: newParentId,
                };
                const newRoot = !prevParentId
                    ? prev.root.filter(id => id !== dragNode.id)
                    : prev.root;
                return {
                    root: newRoot,
                    children: newChildren,
                    parents: newParents,
                    nodes: prev.nodes,
                };
            });
        },
        [tree],
    );

    console.log(treeData);

    return (
        <Tree
            draggable
            showLine
            treeData={treeData}
            onDrop={handleDrop}
            titleRender={renderDepartment}
            dropIndicatorRender={props => {
                const { indent, prefixCls, dropLevelOffset, dropPosition } =
                    props;

                const style: CSSProperties = {
                    height: 2,
                    background: '#4D8EFF',
                    borderRadius: 4,
                    pointerEvents: 'none',
                    position: 'absolute',
                    right: 0,
                };

                switch (dropPosition) {
                    case -1:
                        style.top = 0;
                        style.left = -dropLevelOffset * indent;
                        break;

                    case 1:
                        style.bottom = 0;
                        style.left = -dropLevelOffset * indent;
                        break;

                    case 0:
                        style.bottom = 0;
                        style.left = indent;
                        break;
                }

                return <div className={prefixCls} style={style} />;
            }}
        />
    );
};
