import { AsyncLocalStorage } from 'async_hooks';
import { Container, interfaces } from 'inversify';
import { BaseMiddleware, BaseMiddlewareConstuctor } from './BaseMiddleware';
import { ParseRequestMiddleware } from './ParseRequestMiddleware';
import { RequestLoggerMiddleware } from './RequestLoggerMiddleware';
import { AsyncLocalStorageMiddleware } from './AsyncLocalStorageMiddleware';
import { BaseContext } from './typings/BaseContext';
import { Handler } from './typings/Handler';
import { TYPES } from './typings/TYPES';
import { createLogger } from './logger';

export const createApphostHandlerFactory = (
    setup: (container: interfaces.Container) => void,
) => {
    const rootContainer = new Container();
    rootContainer
        .bind(TYPES.AsyncLocalStorage)
        .toConstantValue(new AsyncLocalStorage());
    rootContainer
        .bind(TYPES.Logger)
        .toConstantValue(createLogger(rootContainer));
    setup(rootContainer);

    return <Context extends BaseContext>(
        ...inputMiddlewares: (
            | BaseMiddlewareConstuctor<Context>
            | Handler<Context>
        )[]
    ) => {
        return async (ctx: Context): Promise<void> => {
            const container = rootContainer.createChild();

            const middlewares: (
                | BaseMiddlewareConstuctor<Context>
                | Handler<Context>
            )[] = [
                AsyncLocalStorageMiddleware,
                ParseRequestMiddleware,
                RequestLoggerMiddleware,
                ...inputMiddlewares,
            ];

            middlewares.forEach(item => {
                if (item.prototype instanceof BaseMiddleware) {
                    container
                        .bind(item)
                        .to(item as BaseMiddlewareConstuctor<Context>);
                }
            });

            const invokeMiddlewares = async (
                middlewares: (
                    | BaseMiddlewareConstuctor<Context>
                    | Handler<Context>
                )[],
            ) => {
                if (!middlewares.length) {
                    return;
                }

                const middleware = middlewares[0];
                const invokeNextMiddleware = async () => {
                    await invokeMiddlewares(middlewares.slice(1));
                };

                if (middleware.prototype instanceof BaseMiddleware) {
                    const middlewareInstance =
                        container.get<BaseMiddleware<Context>>(middleware);

                    middlewareInstance.requestContext = { container, ctx };

                    await middlewareInstance.handler(ctx, invokeNextMiddleware);
                } else {
                    await (middleware as Handler<Context>)(
                        ctx,
                        invokeNextMiddleware,
                    );
                }
            };

            await invokeMiddlewares(middlewares);
        };
    };
};
