import 'reflect-metadata';

import { AsyncLocalStorage } from 'async_hooks';
import { Logger } from '@yandex-int/yandex-logger';
import { Service, ServiceTransport } from '@yandex-int/apphost-lib';
import { AsyncLocalStorageStore, TYPES as ApphostTypes } from '@crm/apphost';
import { Config } from 'services/Config';
import { TVMApiClient, NullTVMApiClient } from 'services/TVMApiClient';
import { TVMCheckMiddleware } from 'utils/TVMCheckMiddleware';
import { TVMServiceTicketMiddleware } from 'utils/TVMServiceTicketMiddleware';
import { createApphostHandlerFactory } from '@crm/apphost';
import { buildSchema } from 'gql/buildSchema';
import { OrganizationResolver } from 'gql/resolvers/OrganizationResolver';
import { DepartmentResolver } from 'gql/resolvers/DepartmentResolver';
import { UserResolver } from 'gql/resolvers/UserResolver';
import { SimpleResolver } from 'gql/resolvers/SimpleResolver';
import { UserService } from 'services/UserService';
import { OrganizationService } from 'services/OrganizationService';
import { DepartmentService, DepartmentMapper } from 'services/Department';
import { HttpClient as IHttpClient } from 'typings/HttpClient';
import { got } from 'got';

import { TYPES } from './typings/TYPES';

import { Gql } from './handlers/gql';
import { Hello } from './handlers/hello';
import { Csrf } from './handlers/csrf';
import * as blackbox from './handlers/blackbox';

const createHandler = createApphostHandlerFactory(container => {
    container.bind(TYPES.Config).to(Config).inSingletonScope();
    container
        .bind(TYPES.OrganizationService)
        .to(OrganizationService)
        .inSingletonScope();
    container.bind(TYPES.UserService).to(UserService).inSingletonScope();
    container
        .bind(TYPES.DepartmentMapper)
        .to(DepartmentMapper)
        .inSingletonScope();
    container
        .bind(TYPES.DepartmentService)
        .to(DepartmentService)
        .inSingletonScope();
    container.bind(UserResolver).toSelf();
    container.bind(OrganizationResolver).toSelf();
    container.bind(DepartmentResolver).toSelf();
    container.bind(SimpleResolver).toSelf();
    container.bind(TYPES.GraphQLSchema).toConstantValue(buildSchema(container));
    const http = got.extend({
        https: {
            certificateAuthority: CA,
        },
    });
    container
        .bind<IHttpClient>(TYPES.HttpClient)
        .toConstantValue(http)
        .whenTargetIsDefault();
    container
        .bind<IHttpClient>(TYPES.HttpClient)
        .toDynamicValue(context => {
            const logger = context.container.get<Logger>(ApphostTypes.Logger);
            const config = context.container.get<Config>(TYPES.Config);

            return http.extend({
                prefixUrl: config.gallifreyUrl,
                hooks: {
                    beforeRequest: [
                        options => {
                            const asyncLocalStorage = context.container.get<
                                AsyncLocalStorage<AsyncLocalStorageStore>
                            >(ApphostTypes.AsyncLocalStorage);
                            const store = asyncLocalStorage.getStore();

                            options.headers['X-Ya-User-Ticket'] =
                                store?.get('tvm_user_ticket');
                            options.headers['X-Ya-Service-Ticket'] = store?.get(
                                'gallifrey_service_ticket',
                            );
                        },
                    ],
                    afterResponse: [
                        response => {
                            const reqId = response.headers['x-yandex-req-id'];
                            const pathname = response.requestUrl.pathname;
                            const method = response.request.options.method;
                            logger.info(
                                'Gallifrey: %s %s. x-yandex-req-id: %s',
                                method,
                                pathname,
                                reqId,
                            );

                            return response;
                        },
                    ],
                },
            });
        })
        .inSingletonScope()
        .whenTargetNamed('gallifrey');
    container.bind(TYPES.TVMClient).toDynamicValue(context => {
        const config = context.container.get<Config>(TYPES.Config);
        const logger = context.container.get<Logger>(ApphostTypes.Logger);
        return config.tvmSelf
            ? new TVMApiClient(
                  {
                      self: config.tvmSelf,
                      token: config.tvmApiToken,
                      serverUrl: config.tvmApiUrl,
                      allowedTVMServiceIds: config.allowedTVMServiceIds,
                  },
                  logger,
              )
            : new NullTVMApiClient();
    });
});

export const createApp = () => {
    const adminPanelHandler = createHandler(TVMCheckMiddleware, Hello);

    return Service.create({ transport: ServiceTransport.GRPC })
        .addHandler('/csrf', createHandler(TVMCheckMiddleware, Csrf))
        .addHandler('/hello', adminPanelHandler)
        .addHandler('/adminpanel', adminPanelHandler)
        .addHandler(
            '/graphql',
            createHandler(TVMCheckMiddleware, TVMServiceTicketMiddleware, Gql),
        )
        .addHandler(
            '/blackbox/request',
            createHandler(TVMCheckMiddleware, blackbox.request),
        )
        .addHandler(
            '/blackbox/response',
            createHandler(TVMCheckMiddleware, blackbox.Response),
        );
};
