import {
    Arg,
    FieldResolver,
    Int,
    Mutation,
    Query,
    Resolver,
    Root,
} from 'type-graphql';
import { inject, injectable } from 'inversify';
import { TYPES } from 'typings/TYPES';
import { UserService } from 'services/UserService';
import { DepartmentService } from 'services/Department';
import {
    CreateDepartmentInput,
    UpdateDepartmentInput,
    User,
    Department,
} from 'gql/schema';

@injectable()
@Resolver(Department)
export class DepartmentResolver {
    public constructor(
        @inject(TYPES.UserService) private userService: UserService,
        @inject(TYPES.DepartmentService)
        private departmentService: DepartmentService,
    ) {}

    @Query(() => [Department])
    public async departments() {
        return this.departmentService.findAll();
    }

    @FieldResolver(() => [User])
    public async users(@Root() department: Department) {
        const userIds = await this.departmentService.findAllDepartmentUsers(
            department.organizationId,
            department.id,
        );

        return await Promise.all(
            userIds.map(userId => this.userService.findOne(userId)),
        );
    }

    @FieldResolver(() => [User])
    public async responsibles(@Root() department: Department) {
        const responsibleIds =
            await this.departmentService.findAllDepartmentResponsibles(
                department.organizationId,
                department.id,
            );

        return await Promise.all(
            responsibleIds.map(id => this.userService.findOne(id)),
        );
    }

    @FieldResolver(() => Int)
    public async usersCount(@Root() department: Department) {
        if (department.usersCount != null) {
            return department.usersCount;
        }

        const departmentWithUsersCount =
            await this.departmentService.findOneOrganizationDepartment(
                department.organizationId,
                department.id,
            );

        return departmentWithUsersCount.usersCount;
    }

    @Mutation(() => Department)
    public async createDepartment(@Arg('input') input: CreateDepartmentInput) {
        return this.departmentService.createDepartment(input);
    }

    @Mutation(() => Department)
    public async updateDepartment(
        @Arg('organizationId', () => Int) organizationId: number,
        @Arg('departmentId') departmentId: string,
        @Arg('input') input: UpdateDepartmentInput,
    ) {
        return this.departmentService.updateDepartment(
            organizationId,
            departmentId,
            input,
        );
    }

    @Mutation(() => Boolean)
    public async archiveDepartment(
        @Arg('organizationId', () => Int) organizationId: number,
        @Arg('departmentId') departmentId: string,
    ) {
        return this.departmentService.archiveDepartment(
            organizationId,
            departmentId,
        );
    }

    @Mutation(() => Boolean)
    public async addUserToDepartment(
        @Arg('organizationId', () => Int) organizationId: number,
        @Arg('departmentId') departmentId: string,
        @Arg('userId', () => Int) userId: number,
    ) {
        return this.departmentService.addUserToDepartment(
            organizationId,
            departmentId,
            userId,
        );
    }

    @Mutation(() => Boolean)
    public async removeUserFromDepartment(
        @Arg('organizationId', () => Int) organizationId: number,
        @Arg('departmentId') departmentId: string,
        @Arg('userId', () => Int) userId: number,
    ) {
        return this.departmentService.removeUserFromDepartment(
            organizationId,
            departmentId,
            userId,
        );
    }

    @Mutation(() => Boolean)
    public async addResponsibleToDepartment(
        @Arg('organizationId', () => Int) organizationId: number,
        @Arg('departmentId') departmentId: string,
        @Arg('userId', () => Int) userId: number,
    ) {
        return this.departmentService.addResponsibleToDepartment(
            organizationId,
            departmentId,
            userId,
        );
    }

    @Mutation(() => Boolean)
    public async removeResponsibleFromDepartment(
        @Arg('organizationId', () => Int) organizationId: number,
        @Arg('departmentId') departmentId: string,
        @Arg('userId', () => Int) userId: number,
    ) {
        return this.departmentService.removeResponsibleFromDepartment(
            organizationId,
            departmentId,
            userId,
        );
    }
}
