import {
    Arg,
    FieldResolver,
    Int,
    Mutation,
    Query,
    Resolver,
    Root,
} from 'type-graphql';
import { inject, injectable } from 'inversify';
import { TYPES } from 'typings/TYPES';
import { OrganizationService } from 'services/OrganizationService';
import { UserService } from 'services/UserService';
import {
    Organization,
    CreateOrganizationInput,
    UpdateOrganizationInput,
    User,
    Department,
    FilterInput,
} from 'gql/schema';
import { DepartmentService } from 'services/Department';

@injectable()
@Resolver(Organization)
export class OrganizationResolver {
    public constructor(
        @inject(TYPES.UserService) private userService: UserService,
        @inject(TYPES.OrganizationService)
        private organizationService: OrganizationService,
        @inject(TYPES.DepartmentService)
        private departmentService: DepartmentService,
    ) {}

    @Query(() => Organization)
    public async organization(@Arg('id', () => Int) id: number) {
        const organization = await this.organizationService.findOne(id);
        if (!organization) {
            throw new Error(`No organization with id: ${id}`);
        }
        return organization;
    }

    @Query(() => [Organization])
    public async organizations() {
        return this.organizationService.findAll();
    }

    @FieldResolver(() => [User])
    public async users(
        @Root() org: Organization,
        @Arg('filter', { nullable: true }) filter?: FilterInput,
    ) {
        const { text } = filter || {};
        const userIds = await this.organizationService.findAllOrganizationUsers(
            org.id,
            {
                text,
            },
        );

        return await Promise.all(
            userIds.map(userId => this.userService.findOne(userId)),
        );
    }

    @FieldResolver(() => [Department])
    public async departments(
        @Root() org: Organization,
        @Arg('filter', { nullable: true }) filter?: FilterInput,
    ) {
        const { text } = filter || {};

        const options =
            text != null
                ? {
                      pattern: {
                          languageCode: 'ru',
                          text,
                      },
                  }
                : undefined;

        return this.departmentService.findAllOrganizationDepartments(
            org.id,
            options,
        );
    }

    @FieldResolver(() => Department)
    public async department(@Root() org: Organization, @Arg('id') id: string) {
        const department =
            await this.departmentService.findOneOrganizationDepartment(
                org.id,
                id,
            );

        if (!department) {
            throw new Error(
                `No department in response, (orgId:${org.id}, departmentId:${id})`,
            );
        }

        return department;
    }

    @Mutation(() => Organization)
    public async createOrganization(
        @Arg('input') input: CreateOrganizationInput,
    ) {
        return this.organizationService.createOrganization(input);
    }

    @Mutation(() => Organization)
    public async updateOrganization(
        @Arg('id', () => Int) id: number,
        @Arg('input') input: UpdateOrganizationInput,
    ) {
        return this.organizationService.updateOrganization(id, input);
    }

    @Mutation(() => Organization)
    public async archiveOrganization(@Arg('id', () => Int) id: number) {
        return this.organizationService.archiveOrganization(id);
    }

    @Mutation(() => Boolean)
    public async addUserToOrganization(
        @Arg('userId', () => Int) userId: number,
        @Arg('organizationId', () => Int) organizationId: number,
    ) {
        return this.organizationService.addUserToOrganization(
            userId,
            organizationId,
        );
    }

    @Mutation(() => Boolean)
    public async removeUserFromOrganization(
        @Arg('userId', () => Int) userId: number,
        @Arg('organizationId', () => Int) organizationId: number,
    ) {
        return this.organizationService.removeUserFromOrganization(
            userId,
            organizationId,
        );
    }
}
