import { injectable, inject } from 'inversify';
import { Logger } from '@yandex-int/yandex-logger';
import {
    OutgoingProtobufJSChunk,
    ProtobufJSContext,
} from '@yandex-int/apphost-lib';
import { BaseMiddleware, TYPES } from '@crm/apphost';
import { hideSecretValue } from 'utils/hideSecretValue';
import { NAppHostHttp, NAppHostTvmUserTicket, NBlackbox } from '@crm/protos';

const { THttpResponse } = NAppHostHttp;

@injectable()
export class Response extends BaseMiddleware<ProtobufJSContext> {
    constructor(@inject(TYPES.Logger) private logger: Logger) {
        super();
    }

    public async handler(ctx: ProtobufJSContext) {
        const data = await ctx.allIncomingChunks();

        const blackboxHttpResponse = data
            .getOnlyItem('blackbox_http_response')
            .parseProto(THttpResponse);

        const blackboxResponse = JSON.parse(
            String(blackboxHttpResponse.Content),
        );

        this.logger.info(
            { status: blackboxResponse.status, error: blackboxResponse.error },
            'blackbox response',
        );

        const output = new OutgoingProtobufJSChunk();

        if (blackboxResponse.user_ticket) {
            const tvmUserTicket =
                NAppHostTvmUserTicket.TTvmUserTicket.fromObject({
                    UserTicket: blackboxResponse.user_ticket,
                });
            output.addProtoItem(
                'tvm_user_ticket',
                NAppHostTvmUserTicket.TTvmUserTicket,
                tvmUserTicket,
            );
            this.logger.info(
                'Add tvm_user_ticket item with %s',
                hideSecretValue(blackboxResponse.user_ticket),
            );
        }

        if (blackboxResponse.login || blackboxResponse.uid) {
            const user = NBlackbox.TUser.fromObject({
                Login: blackboxResponse.login,
                Uid: blackboxResponse.uid,
            });
            output.addProtoItem('blackbox_user', NBlackbox.TUser, user);
        }

        ctx.send(output);
    }
}
