import {
    IncomingProtobufJSChunk,
    ProtobufJSContext,
} from '@yandex-int/apphost-lib';
import { inject, injectable } from 'inversify';
import { BaseMiddleware } from '@crm/apphost';
import { TYPES } from 'typings/TYPES';
import cookie from 'cookie';
import { v2 } from '@yandex-int/secret-key';
import { Config } from 'services/Config';
import { NAppHostHttp, NBlackbox } from '@crm/protos';
import { CSRF_COOKIE_KEY, CSRF_HEADER_KEY, LIFETIME } from './constants';

const { THttpRequest, THttpResponse } = NAppHostHttp;
const EMethod = THttpRequest.EMethod;

@injectable()
export class Csrf extends BaseMiddleware<ProtobufJSContext> {
    constructor(@inject(TYPES.Config) private config: Config) {
        super();
    }

    public parseCookies(
        data: IncomingProtobufJSChunk,
    ): Record<string, string | undefined> {
        const httpRequest = data
            .getOnlyItem('proto_http_request')
            .parseProto(THttpRequest);
        const cookieHeader = httpRequest.Headers.find(
            header => String(header.Name).toLowerCase() === 'cookie',
        );

        if (!cookieHeader || !cookieHeader.Value) {
            return {};
        }

        return cookie.parse(cookieHeader.Value);
    }

    public parseUid(data: IncomingProtobufJSChunk): string {
        const blackboxUserItem = data.getFirstItem('blackbox_user');
        const blackboxUser =
            blackboxUserItem && blackboxUserItem.parseProto(NBlackbox.TUser);
        if (!blackboxUser || !blackboxUser.Uid) {
            return '0';
        }

        return blackboxUser.Uid;
    }

    public generateToken(data: IncomingProtobufJSChunk): string {
        const uid = this.parseUid(data);
        const cookies = this.parseCookies(data);
        const yandexuid = cookies.yandexuid;
        let opts = {
            uid: uid,
            yandexuid: yandexuid,
            salt: this.config.appKey,
            timestamp: Date.now(),
        };

        if (!yandexuid) {
            return '';
        }

        return v2(opts);
    }

    public isTokenValid(data: IncomingProtobufJSChunk): boolean {
        const httpRequest = data
            .getOnlyItem('proto_http_request')
            .parseProto(THttpRequest);

        if (
            [EMethod.Get, EMethod.Options, EMethod.Head].includes(
                httpRequest.Method,
            )
        ) {
            return true;
        }

        const csrfHeader = httpRequest.Headers.find(
            header => String(header.Name).toLowerCase() === CSRF_HEADER_KEY,
        );
        const token = csrfHeader?.Value || '';
        const cookies = this.parseCookies(data);
        const yandexuid = cookies.yandexuid;
        const uid = this.parseUid(data);
        const opts = {
            uid: uid,
            yandexuid: yandexuid,
            salt: this.config.appKey,
            lifetime: LIFETIME,
        };

        return Boolean(token) && Boolean(yandexuid) && v2.isValid(token, opts);
    }

    public async handler(ctx: ProtobufJSContext) {
        const data = await ctx.allIncomingChunks();
        const isValid = this.isTokenValid(data);
        const newToken = this.generateToken(data);

        const Headers: NAppHostHttp.ITHeader[] = [
            { Name: 'Set-Cookie', Value: `${CSRF_COOKIE_KEY}=${newToken}` },
        ];

        const csrfHttpResponseProto = THttpResponse.fromObject({
            Content: isValid ? undefined : Buffer.from('Invalid CSRF token'),
            Headers,
            StatusCode: isValid ? undefined : 403,
        });
        isValid && ctx.sendFlag('csrf-valid');
        ctx.sendProtoItem(
            'proto_http_response',
            THttpResponse,
            csrfHttpResponseProto,
        );
    }
}
