import { AsyncLocalStorage } from 'async_hooks';
import { ProtobufJSContext } from '@yandex-int/apphost-lib';
import { graphql, GraphQLSchema } from 'graphql';
import { NAppHostHttp, NAppHostTvmUserTicket, NBlackbox } from '@crm/protos';
import {
    AsyncLocalStorageStore,
    BaseMiddleware,
    TYPES as ApphostTypes,
} from '@crm/apphost';
import { inject, injectable } from 'inversify';
import { TYPES } from 'typings/TYPES';
import index from './index.html';

const { THttpResponse, THttpRequest } = NAppHostHttp;

@injectable()
export class Gql extends BaseMiddleware<ProtobufJSContext> {
    public constructor(
        @inject(TYPES.GraphQLSchema) private schema: GraphQLSchema,
        @inject(ApphostTypes.AsyncLocalStorage)
        private asyncLocalStorage: AsyncLocalStorage<AsyncLocalStorageStore>,
    ) {
        super();
    }

    private setResponseWithIndexHtml(ctx: ProtobufJSContext) {
        const httpResponseProto = THttpResponse.fromObject({
            Content: Buffer.from(index),
            Headers: [
                { Name: 'Content-Type', Value: 'text/html; charset=UTF-8' },
            ],
            StatusCode: 200,
        });

        ctx.sendProtoItem(
            'proto_http_response',
            THttpResponse,
            httpResponseProto,
        );
    }

    public async handler(ctx: ProtobufJSContext): Promise<void> {
        const data = await ctx.allIncomingChunks();

        const httpRequest = data
            .getOnlyItem('proto_http_request')
            .parseProto(THttpRequest);

        if (httpRequest.Method === THttpRequest.EMethod.Get) {
            this.setResponseWithIndexHtml(ctx);
            return;
        }

        const tvmUserTicket = data
            .getOnlyItem('tvm_user_ticket')
            .parseProto(NAppHostTvmUserTicket.TTvmUserTicket).UserTicket;

        const store = this.asyncLocalStorage.getStore();
        store?.set('tvm_user_ticket', tvmUserTicket);

        // TODO: handle parse error
        const graphqlRequestData = JSON.parse(String(httpRequest.Content));

        const response = await graphql({
            schema: this.schema,
            rootValue: {
                me: () => {
                    const user = data
                        .getOnlyItem('blackbox_user')
                        .parseProto(NBlackbox.TUser);

                    return user.Login;
                },
            },
            source: graphqlRequestData.query,
            variableValues: graphqlRequestData.variables,
            operationName: graphqlRequestData.operationName,
        });

        const httpResponseProto = THttpResponse.fromObject({
            Content: Buffer.from(JSON.stringify(response)),
            Headers: [
                {
                    Name: 'Content-Type',
                    Value: 'application/json; charset=UTF-8',
                },
            ],
            StatusCode: 200,
        });

        ctx.sendProtoItem(
            'proto_http_response',
            THttpResponse,
            httpResponseProto,
        );
    }
}
