import { departmentmanager } from '@crm/protos';
import { injectable } from 'inversify';
import { DepartmentDto } from './dto/DepartmentDto';
import { NameDto } from './dto/NameDto';

interface MapperOptions {
    includeUsersCount?: boolean;
}

@injectable()
export class DepartmentMapper {
    private createDto(
        payload: {
            department: departmentmanager.IDepartment;
            usersCount?: number;
        },
        options?: MapperOptions,
    ): DepartmentDto {
        const { includeUsersCount = false } = options || {};
        const { department, usersCount } = payload;

        const data = department?.data;
        const organizationId = Number(data?.organizationId);
        const id = department?.id as string;

        const dto: DepartmentDto = {
            id,
            parentDepartmentId: data?.parentDepartmentId as string,
            organizationId,
            names: data?.names as NameDto[],
        };

        if (includeUsersCount) {
            dto.usersCount = usersCount;
        }

        return dto;
    }

    public toDto(
        response: departmentmanager.IDepartmentsResponse,
        options?: MapperOptions,
    ): DepartmentDto {
        const department = response
            .departments?.[0] as departmentmanager.IDepartment;
        const usersCount = response?.usersCount?.[0]?.usersCount as
            | number
            | undefined;

        return this.createDto(
            {
                department,
                usersCount,
            },
            options,
        );
    }

    public toDtoArray(
        response: departmentmanager.IDepartmentsResponse,
        options?: MapperOptions,
    ): DepartmentDto[] {
        const map = new Map<
            string,
            {
                department: departmentmanager.IDepartment;
                usersCount?: number;
            }
        >();

        const departmentsLength = response?.departments?.length || 0;
        for (let i = 0; i < departmentsLength; i++) {
            const department = response?.departments?.[
                i
            ] as departmentmanager.IDepartment;
            const orgId = department.data?.organizationId as number;
            const departmentId = department.id as string;

            const key = `${orgId}:${departmentId}`;
            map.set(key, {
                department,
            });
        }

        if (options?.includeUsersCount) {
            const usersCountLength = response?.usersCount?.length || 0;
            for (let i = 0; i < usersCountLength; i++) {
                const usersCount = response?.usersCount?.[
                    i
                ] as departmentmanager.IDepartmentUsersCount;
                const orgId = usersCount.organizationId as number;
                const departmentId = usersCount.departmentId as string;

                const key = `${orgId}:${departmentId}`;
                const obj = map.get(key);
                if (obj) {
                    obj.usersCount = usersCount.usersCount as number;
                }
            }
        }

        return Array.from(map).map(([_, { department, usersCount }]) =>
            this.createDto(
                {
                    department,
                    usersCount,
                },
                options,
            ),
        );
    }
}
