import { departmentmanager } from '@crm/protos';
import { inject, injectable, named } from 'inversify';
import { HttpClient } from 'typings/HttpClient';
import { TYPES } from 'typings/TYPES';
import { DepartmentMapper } from './DepartmentMapper';
import { CreateDepartmentDto } from './dto/CreateDepartmentDto';
import { UpdateDepartmentDto } from './dto/UpdateDepartmentDto';

@injectable()
export class DepartmentService {
    public constructor(
        @inject(TYPES.HttpClient)
        @named('gallifrey')
        private http: HttpClient,
        @inject(TYPES.DepartmentMapper) private mapper: DepartmentMapper,
    ) {}

    public findAll() {
        return this.http.get('department/list').then(protoResponse => {
            const response = departmentmanager.DepartmentsResponse.decode(
                protoResponse.rawBody,
            );
            return this.mapper.toDtoArray(response);
        });
    }

    public findAllOrganizationDepartments(
        orgId: number,
        options: {
            pattern?: {
                text: string;
                languageCode: string;
            };
        } = {},
    ) {
        const includeUsersCount = true;
        const { pattern } = options;

        return this.http
            .get(`department/${orgId}/list`, {
                searchParams: {
                    includeUsersCount,
                    pattern: pattern?.text,
                    languageCode: pattern?.languageCode,
                },
            })
            .then(protoResponse => {
                const response = departmentmanager.DepartmentsResponse.decode(
                    protoResponse.rawBody,
                );
                return this.mapper.toDtoArray(response, {
                    includeUsersCount,
                });
            });
    }

    public findOneOrganizationDepartment(orgId: number, departmentId: string) {
        const includeUsersCount = true;
        return this.http
            .get(`department/${orgId}/${departmentId}`, {
                searchParams: {
                    includeUsersCount: includeUsersCount,
                },
            })
            .then(protoResponse => {
                const response = departmentmanager.DepartmentsResponse.decode(
                    protoResponse.rawBody,
                );

                return this.mapper.toDto(response, {
                    includeUsersCount,
                });
            });
    }

    public findAllDepartmentUsers(orgId: number, departmentId: string) {
        return this.http
            .get(`department/${orgId}/${departmentId}/users`)
            .then(response => {
                const usersId =
                    departmentmanager.DepartmentUsersResponse.decode(
                        response.rawBody,
                    ).usersId;

                return usersId || [];
            });
    }

    public findAllDepartmentResponsibles(
        orgId: number,
        departmentId: string,
        options: {
            includeParentDepartments?: boolean;
        } = {},
    ) {
        const { includeParentDepartments } = options;

        return this.http
            .get(`department/${orgId}/${departmentId}/responsible`, {
                searchParams: {
                    includeParentDepartments,
                },
            })
            .then(response => {
                const usersId =
                    departmentmanager.DepartmentUsersResponse.decode(
                        response.rawBody,
                    ).usersId;

                return usersId || [];
            });
    }

    public createDepartment(dto: CreateDepartmentDto) {
        return this.http
            .patch(`department/create`, {
                body: Buffer.from(
                    departmentmanager.CreateDepartmentRequest.encode({
                        departmentData: {
                            organizationId: dto.organizationId,
                            parentDepartmentId: dto.parentDepartmentId || null,
                            names: dto.names,
                        },
                        departmentResponsibleUserIds: dto.responsibleIds,
                    }).finish(),
                ),
            })
            .then(protoResponse => {
                const response = departmentmanager.DepartmentsResponse.decode(
                    protoResponse.rawBody,
                );

                return this.mapper.toDto(response);
            });
    }

    public async updateDepartment(
        orgId: number,
        departmentId: string,
        dto: UpdateDepartmentDto,
    ) {
        const existingDepartment = await this.findOneOrganizationDepartment(
            orgId,
            departmentId,
        );

        if (!existingDepartment) {
            throw new Error(
                `No department to update with orgId: ${orgId}, departmentId: ${departmentId}`,
            );
        }

        return this.http
            .put(`department/${orgId}/${departmentId}/update`, {
                body: Buffer.from(
                    departmentmanager.UpdateDepartmentData.encode({
                        newOrganizationId: dto.organizationId,
                        newParentDepartmentId: dto.parentDepartmentId,
                        // TODO https://st.yandex-team.ru/CRM-18909
                        newNames: dto.names
                            ? dto.names
                            : existingDepartment.names,
                    }).finish(),
                ),
            })
            .then(protoResponse => {
                const response = departmentmanager.DepartmentsResponse.decode(
                    protoResponse.rawBody,
                );

                return this.mapper.toDto(response);
            });
    }

    public addResponsibleToDepartment(
        orgId: number,
        departmentId: string,
        userId: number,
    ) {
        return this.http
            .put(
                `department/${orgId}/${departmentId}/add/responsible/${userId}`,
            )
            .then(() => true)
            .catch(() => false);
    }

    public removeResponsibleFromDepartment(
        orgId: number,
        departmentId: string,
        userId: number,
    ) {
        return this.http
            .delete(
                `department/${orgId}/${departmentId}/remove/responsible/${userId}`,
            )
            .then(() => true)
            .catch(() => false);
    }

    public archiveDepartment(orgId: number, departmentId: string) {
        return this.http
            .post(`department/${orgId}/${departmentId}/archive`)
            .then(() => true)
            .catch(() => false);
    }

    public addUserToDepartment(
        orgId: number,
        departmentId: string,
        userId: number,
    ) {
        return this.http
            .put(`department/${orgId}/${departmentId}/add/user/${userId}`)
            .then(() => true)
            .catch(() => false);
    }

    public removeUserFromDepartment(
        orgId: number,
        departmentId: string,
        userId: number,
    ) {
        return this.http
            .delete(`department/${orgId}/${departmentId}/remove/user/${userId}`)
            .then(() => true)
            .catch(() => false);
    }
}
