import { organizationmanager, usermanager } from '@crm/protos';
import { inject, injectable, named } from 'inversify';
import { HttpClient } from 'typings/HttpClient';
import { TYPES } from 'typings/TYPES';

export interface OrganizationDto {
    id: number;
    pool?: number | null;
    slug?: string | null;
    name?: string | null;
}
export type CreateOrganizationDto = Omit<OrganizationDto, 'id'>;
export type UpdateOrganizationDto = CreateOrganizationDto;

@injectable()
export class OrganizationService {
    public constructor(
        @inject(TYPES.HttpClient)
        @named('gallifrey')
        private http: HttpClient,
    ) {}

    private toDto(
        protoOrg: organizationmanager.IOrganization,
    ): OrganizationDto {
        return {
            id: Number(protoOrg.id?.value),
            pool: Number(protoOrg.data?.pool),
            slug: protoOrg.data?.slug,
            name: protoOrg.data?.name,
        };
    }

    public findAllOrganizationUsers(
        id: number,
        options: {
            text?: string;
        } = {},
    ) {
        const { text } = options;

        return this.http
            .get(`user/organization/list/${id}`, {
                searchParams: {
                    pattern: text,
                },
            })
            .then(
                response =>
                    usermanager.OrganizationUsersResponse.decode(
                        response.rawBody,
                    ).usersId || [],
            );
    }

    public findOne(id: number): Promise<OrganizationDto | null> {
        return this.http.get(`organization/${id}`).then(response => {
            const org = organizationmanager.OrganizationResponse.decode(
                response.rawBody,
            ).organization;

            if (!org) {
                return null;
            }

            return this.toDto(org);
        });
    }

    public findAll(): Promise<OrganizationDto[]> {
        return this.http.get('organizations/list').then(response => {
            const organizations =
                organizationmanager.OrganizationsResponse.decode(
                    response.rawBody,
                ).organizations || [];
            return organizations.map(this.toDto);
        });
    }

    public createOrganization(dto: CreateOrganizationDto) {
        return this.http
            .put(`organization/create`, {
                body: Buffer.from(
                    organizationmanager.OrganizationData.encode({
                        pool: dto.pool,
                        slug: dto.slug,
                        name: dto.name,
                    }).finish(),
                ),
            })
            .then(response => {
                const proto = organizationmanager.OrganizationResponse.decode(
                    response.rawBody,
                ).organization;

                if (!proto) {
                    throw new Error('No organization in response');
                }

                return this.toDto(proto);
            });
    }

    public updateOrganization(id: number, dto: UpdateOrganizationDto) {
        return this.http
            .patch(`organization/update/${id}`, {
                body: Buffer.from(
                    organizationmanager.UpdateOrganizationData.encode({
                        newPool: dto.pool,
                        newSlug: dto.slug,
                        newName: dto.name,
                    }).finish(),
                ),
            })
            .then(response => {
                const proto = organizationmanager.OrganizationResponse.decode(
                    response.rawBody,
                ).organization;

                if (!proto) {
                    throw new Error('No organization in response');
                }

                return this.toDto(proto);
            });
    }

    public archiveOrganization(id: number) {
        return this.http.post(`organization/archive/${id}`).then(response => {
            const proto = organizationmanager.OrganizationResponse.decode(
                response.rawBody,
            ).organization;

            if (!proto) {
                throw new Error('No organization in response');
            }

            return this.toDto(proto);
        });
    }

    public addUserToOrganization(userId: number, orgId: number) {
        return this.http
            .put(`user/organization/add/${orgId}/${userId}`)
            .then(() => true)
            .catch(() => false);
    }

    public removeUserFromOrganization(userId: number, orgId: number) {
        return this.http
            .delete(`user/organization/remove/${orgId}/${userId}`)
            .then(() => true)
            .catch(() => false);
    }
}
