// https://wiki.yandex-team.ru/passport/tvm2/tvm-daemon#apiv2
import { injectable } from 'inversify';
import got, { Got, RequestError } from 'got';
import {
    TVMApiClientOptions,
    TVMApiClientGetTicketsOptions,
    TVMServiceTicketResponse,
    TVMServicePingResponse,
    TVMTicketsCheckResponse,
    TVMServiceTicketCheck,
    TVMApiClient as ITVMApiClient,
} from 'typings/TVMApiClient';
import { Logger } from '@yandex-int/yandex-logger';
import { hideSecretValue } from 'utils/hideSecretValue';
import { TVMApiCheckServiceTicketError } from './TVMApiCheckServiceTicketError';

@injectable()
export class TVMApiClient implements ITVMApiClient {
    private readonly tvmApiClient: Got;

    constructor(
        private readonly options: TVMApiClientOptions,
        private logger?: Logger,
    ) {
        this.tvmApiClient = got.extend({
            prefixUrl: options.serverUrl,
            responseType: 'json',
            headers: { Authorization: options.token },
            searchParams: { self: this.options.self },
        });
    }

    ping() {
        return this.tvmApiClient.get<TVMServicePingResponse>('ping');
    }

    checkServiceTicket(serviceTicket: string): Promise<TVMServiceTicketCheck> {
        this.logger?.info(
            'Check service ticket: %s',
            hideSecretValue(serviceTicket),
        );

        return this.tvmApiClient
            .get<TVMTicketsCheckResponse>('v2/check', {
                headers: {
                    'X-Ya-Service-Ticket': serviceTicket,
                },
            })
            .catch((error: RequestError) => {
                this.logger?.info(error, 'Check request failed');
                const body = error.response?.body as TVMTicketsCheckResponse;

                throw new TVMApiCheckServiceTicketError(body.service);
            })
            .then(response => {
                if (
                    this.options.allowedTVMServiceIds &&
                    this.options.allowedTVMServiceIds.length &&
                    !this.options.allowedTVMServiceIds.includes(
                        response.body.service.src,
                    )
                ) {
                    throw new TVMApiCheckServiceTicketError({
                        ...response.body.service,
                        status: 'INVALID_SRC',
                        error: `Wrong ticket src, expected ${this.options.allowedTVMServiceIds}; got ${response.body.service.src}`,
                    });
                }

                return response.body.service;
            });
    }

    async getTickets(options: TVMApiClientGetTicketsOptions) {
        const response = await this.tvmApiClient.get<
            Record<string, TVMServiceTicketResponse | undefined>
        >('v2/tickets', {
            searchParams: { dsts: String(options.dsts) },
        });

        return response.body;
    }
}
