import { usermanager } from '@crm/protos';
import { inject, injectable, named } from 'inversify';
import { HttpClient } from 'typings/HttpClient';
import { TYPES } from 'typings/TYPES';

export interface UserDto {
    id: number;
    firstName?: string;
    lastName?: string;
}
export type CreateUserDto = Omit<UserDto, 'id'>;
export type UpdateUserDto = Partial<CreateUserDto>;

@injectable()
export class UserService {
    public constructor(
        @inject(TYPES.HttpClient)
        @named('gallifrey')
        private http: HttpClient,
    ) {}

    private toDto(protoUser: usermanager.IUser): UserDto {
        return {
            id: Number(protoUser.userId),
            firstName: protoUser.data?.firstName || undefined,
            lastName: protoUser.data?.secondName || undefined,
        };
    }

    public findOne(id: number): Promise<UserDto | null> {
        return this.http.get(`user/${id}`).then(response => {
            const proto = usermanager.UserResponse.decode(
                response.rawBody,
            ).user;

            if (!proto) {
                return null;
            }

            return this.toDto(proto);
        });
    }

    public findAll() {
        return this.http.get('users/list').then(response => {
            const protos =
                usermanager.UsersResponse.decode(response.rawBody).users || [];
            return protos.map(this.toDto);
        });
    }

    public createUser(dto: CreateUserDto) {
        return this.http
            .put(`user/create`, {
                body: Buffer.from(
                    usermanager.UserData.encode({
                        firstName: dto.firstName,
                        secondName: dto.lastName,
                    }).finish(),
                ),
            })
            .then(response => {
                const proto = usermanager.UserResponse.decode(
                    response.rawBody,
                ).user;

                if (!proto) {
                    throw new Error('No user in response');
                }

                return this.toDto(proto);
            });
    }

    public updateUser(id: number, dto: UpdateUserDto) {
        return this.http
            .patch(`user/${id}/update`, {
                body: Buffer.from(
                    usermanager.UpdateUserData.encode({
                        newFirstName: dto.firstName,
                        newSecondName: dto.lastName,
                    }).finish(),
                ),
            })
            .then(response => {
                const proto = usermanager.UserResponse.decode(
                    response.rawBody,
                ).user;

                if (!proto) {
                    throw new Error('No user in response');
                }

                return this.toDto(proto);
            });
    }

    public archiveUser(id: number) {
        return this.http.post(`user/${id}/archive`).then(response => {
            const proto = usermanager.UserResponse.decode(
                response.rawBody,
            ).user;

            if (!proto) {
                throw new Error('No user in response');
            }

            return this.toDto(proto);
        });
    }
}
