import React, {
    FC,
    useEffect,
    useRef,
    useCallback,
    MutableRefObject,
} from "react";
import { useKeyboard } from "../../../useKeyboard";
import { PopupMenu } from "../../../PopupMenu";
import { KeyValueReading } from "../../components/KeyValueReading";
import { GridTemplate } from "../../components/GridTemplate";
import { EditingArea } from "../../components/EditingArea";
import { useListLoad } from "../../utils/useListLoad";
import { selectOptionToItem } from "../../utils/selectOptionToItem";
import { SelectEditingProps } from "../Select.types";

export const Editing: FC<SelectEditingProps> = (props) => {
    const {
        tabIndex,
        editingRef,
        display = false,
        label,
        name,
        editingValue,
        onLoad,
        isEditingLoading,
        onEditingChange,
        onChange,
        onEvent,
    } = props;

    const {
        list: options,
        handleLoad,
        isLoading,
    } = useListLoad({
        onLoad,
    });
    const editingAreaRef = useRef<HTMLDivElement>(null);

    const handleClose = () => {
        if (!display) {
            return;
        }

        onEditingChange?.(false);
    };

    useKeyboard(
        {
            onEsc: () => onEditingChange?.(false),
            onTab: (event) => event.preventDefault(),
        },
        display,
        [onEditingChange]
    );

    useEffect(() => {
        handleLoad();
    }, [display]);

    const handleInnerRef = useCallback(
        (instance: HTMLDivElement) => {
            editingRef?.(instance);
            (editingAreaRef as MutableRefObject<HTMLDivElement>).current =
                instance;
        },
        [editingRef]
    );

    const handleChange = (value: number | string) => {
        onChange?.(options.find((option) => option.id === value)!);
    };

    return (
        <EditingArea
            name={name}
            display={display}
            innerRef={handleInnerRef}
            tabIndex={tabIndex}
        >
            <GridTemplate>
                <KeyValueReading
                    label={label}
                    readingValue={editingValue && editingValue.text}
                    discardReadingArea
                />
            </GridTemplate>

            <PopupMenu
                isLoading={isEditingLoading || isLoading}
                isVisible={display}
                anchor={editingAreaRef.current}
                items={options.map(selectOptionToItem)}
                onChange={handleChange}
                onClose={handleClose}
                isFocused={display}
                onEvent={onEvent}
            />
        </EditingArea>
    );
};
