import React, {
    useState,
    useEffect,
    useRef,
    useCallback,
    useMemo,
} from "react";
import debounce from "lodash/debounce";
import { Instance } from "@popperjs/core";
import { useKeyboard } from "../../../useKeyboard";
import { BubbleProps, Bubble } from "../../../Bubble";
import { ItemSimple } from "@yandex-lego/components/Menu";
import { Text } from "../../../lego2/Text";
import isPromise from "../../../isPromise";
import { BubbleInput } from "../../../BubbleInput";
import { PopupMenu } from "../../../PopupMenu";
import { useListLoad } from "../../utils/useListLoad";
import { EditingArea } from "../../components/EditingArea";
import { SelectOption } from "../../types/SelectOption";
import {
    SuggestSelectMultipleEditingProps,
    SuggestSelectMultipleEditingValue,
} from "../SuggestSelectMultiple.types";
import { LOAD_DEBOUNCE_DELAY } from "./Editing.config";

const getOptionId = <T extends {}>(item: T) =>
    (item as unknown as SelectOption).id;
const renderPopupOption = <T extends {}>(item: T) =>
    (item as unknown as SelectOption).text;
const renderSelectedOption = <T extends {}>(item: T, props: BubbleProps) => (
    <Bubble canDelete {...props}>
        <Text typography="control-m">
            {(item as unknown as SelectOption).text}
        </Text>
    </Bubble>
);

export const Editing = <T extends {}>(
    props: SuggestSelectMultipleEditingProps<T>
) => {
    const {
        tabIndex,
        editingRef,
        display = false,
        label,
        name,
        editingValue,
        onLoad,
        isEditingLoading,
        onEditingChange,
        onChange,
        getKey = getOptionId,
        renderPopupItem = renderPopupOption,
        renderSelectedBubble = renderSelectedOption,
        appendAddon,
        popperRef: outerPopperRef,
        instanceRef,
        renderPopupMenu,
        onEvent,
        canListenKeyboard = true,
        canFocusMenu = true,
        isContentEditableDisabled = false,
        canDelete = true,
    } = props;

    const canFocusPopup = useRef(false);
    const popperRef = useRef<Instance>();
    const deletedMapRef = useRef<Record<string | number, boolean>>({});
    const [isMenuFocusedState, setMenuFocusedState] = useState(true);
    const [loadingKeys, setLoadingKeys] = useState<
        Record<number | string, boolean>
    >({});
    const [text, setText] = useState("");
    const {
        list: options,
        isLoading,
        handleLoad,
    } = useListLoad({
        onLoad,
    });
    const editingAreaRef = useRef<HTMLDivElement | null>();
    const contentEditableRef = useRef<HTMLDivElement>();

    const emitChange = useCallback(
        (value: SuggestSelectMultipleEditingValue<T>) => {
            const possiblyPromise = onChange?.(value);
            if (isPromise(possiblyPromise)) {
                possiblyPromise.then(() => {
                    popperRef.current?.update();
                });
                return possiblyPromise;
            }

            popperRef.current?.update();
        },
        [onChange]
    );
    const debouncedLoad = useCallback(
        debounce(handleLoad, LOAD_DEBOUNCE_DELAY),
        []
    );

    const emitDelete = useCallback(
        async (value: T) => {
            const key = getKey(value);
            if (deletedMapRef.current[key]) {
                return;
            }

            try {
                deletedMapRef.current[key] = true;
                await emitChange(
                    editingValue.filter((item) => getKey(item) !== key)
                );
            } finally {
                deletedMapRef.current[key] = false;
            }
        },
        [emitChange, editingValue]
    );

    const handleClose = useCallback(() => {
        if (!display) {
            return;
        }

        onEditingChange?.(false);
    }, [onEditingChange, display]);

    useEffect(() => {
        if (display) {
            handleLoad(text);
        } else {
            setText("");
        }
        canFocusPopup.current = display;
    }, [display]);

    const handleContentEditableRef = useCallback(
        (instance: HTMLDivElement) => {
            editingRef?.(instance);
            contentEditableRef.current = instance;
        },
        [editingRef]
    );

    const canUseKeyboard = display && canListenKeyboard;
    useKeyboard(
        {
            onSpaceCapture: (event) => event.stopImmediatePropagation(),
            onEsc: () => onEditingChange?.(false),
        },
        canUseKeyboard,
        [onEditingChange]
    );

    const handlePopupMenuChange = useCallback(
        async (value: number | string | T) => {
            contentEditableRef.current?.focus();
            const newOptionId =
                typeof value === "object"
                    ? getKey(value)
                    : (value as number | string);
            const newOption =
                typeof value === "object"
                    ? value
                    : options.find((option) => getKey(option) === newOptionId)!;

            if (
                editingValue.find(
                    (option) => getKey(option) === getKey(newOption)
                )
            ) {
                return;
            }

            setLoadingKeys((prev) => ({ ...prev, [newOptionId]: true }));
            try {
                await emitChange([...editingValue, newOption]);
                setText((prevText) => {
                    if (prevText !== "") {
                        handleLoad("");
                    }
                    return "";
                });
            } finally {
                setLoadingKeys((prev) => ({ ...prev, [newOptionId]: false }));
            }
        },
        [options, editingValue, emitChange]
    );

    const handleTextinputChange = useCallback((value: string) => {
        popperRef.current?.update();
        setText(value);
        debouncedLoad(value);
    }, []);

    useEffect(() => {
        instanceRef?.({
            reload: () => {
                handleLoad("");
            },
            resetText: () => setText(""),
        });
    }, [instanceRef, handleLoad, text]);

    const handlePopperRef = useCallback(
        (instance: Instance) => {
            popperRef.current = instance;
            outerPopperRef?.(instance);
        },
        [outerPopperRef]
    );

    const popupMenuItems = useMemo(() => {
        const items: ItemSimple[] = [];
        for (let i = 0; i < options.length; i++) {
            const option = options[i];
            if (editingValue.find((item) => getKey(item) === getKey(option))) {
                continue;
            }
            const key = getKey(option);
            items.push({
                disabled: loadingKeys[key],
                value: key,
                content: renderPopupItem(option),
            });
        }

        return items;
    }, [options, editingValue, loadingKeys]);

    const isValueLengthwise =
        Array.isArray(editingValue) && editingValue.length > 0;
    const isMenuFocused =
        canFocusMenu && canFocusPopup.current && isMenuFocusedState;
    const isMenuLoading = isEditingLoading || isLoading;

    return (
        <EditingArea
            name={name}
            display={display}
            tabIndex={-1}
            innerRef={(instance) => (editingAreaRef.current = instance)}
        >
            <BubbleInput<T>
                display={display}
                bubbles={editingValue}
                onBubbleDelete={emitDelete}
                renderBubble={renderSelectedBubble}
                onFocusedIndexChange={useCallback(
                    (index) => setMenuFocusedState(index === -1),
                    []
                )}
                getKey={getKey}
                canUseKeyboard={canUseKeyboard}
                canDelete={canDelete}
                tabIndex={tabIndex}
                text={text}
                placeholder={!isValueLengthwise ? label : undefined}
                onTextChange={handleTextinputChange}
                contentEditableRef={handleContentEditableRef}
                hasClear={false}
                isContentEditableDisabled={isContentEditableDisabled}
            />

            {renderPopupMenu &&
                renderPopupMenu({
                    text,
                    items: options,
                    isLoading: isMenuLoading,
                    onClose: handleClose,
                    isVisible: display,
                    anchor: editingAreaRef.current,
                    isFocused: isMenuFocused,
                    onChange: handlePopupMenuChange,
                    popperRef: handlePopperRef,
                    onEvent,
                })}
            {!renderPopupMenu && (
                <PopupMenu
                    onClose={handleClose}
                    isLoading={isMenuLoading}
                    isVisible={display}
                    anchor={editingAreaRef.current}
                    items={popupMenuItems}
                    onChange={handlePopupMenuChange}
                    isFocused={isMenuFocused}
                    popperRef={handlePopperRef}
                    onEvent={onEvent}
                />
            )}

            {appendAddon}
        </EditingArea>
    );
};
