import React, {
    FC,
    useState,
    useRef,
    useCallback,
    useEffect,
    memo,
} from "react";
import { useOutsideClick } from "../../../useOutsideClick";
import { useKeyboard } from "../../../useKeyboard";
import { BubbleInput } from "../../../BubbleInput";
import { EditingArea } from "../../components/EditingArea";
import {
    TextinputEditingProps,
    TextinputEditingValue,
} from "../Textinput.types";

export const Editing: FC<TextinputEditingProps> = memo((props) => {
    const {
        label,
        name,
        editingValue,
        onChange,
        onEditingChange,
        hasClear = true,
        autoFocus = true,
        display,
        editingRef,
        tabIndex,
        isMultiline,
    } = props;

    const [value, setValue] = useState(editingValue);
    const textinputRef = useRef<HTMLDivElement>();
    const contentEditableRef = useRef<HTMLElement>();

    const emitChange = useCallback(
        (newValue: TextinputEditingValue) => {
            if (editingValue === newValue) {
                onEditingChange?.(false);
                return;
            }

            onChange?.(newValue);
        },
        [editingValue, onChange, onEditingChange]
    );

    useOutsideClick(
        textinputRef,
        useCallback(() => {
            if (!display) {
                return;
            }

            emitChange(value);
        }, [value, emitChange, display]),
        true
    );

    useEffect(() => {
        setValue(editingValue);
    }, [editingValue]);

    useEffect(() => {
        if (!display) {
            setValue(editingValue);
        }
    }, [display]);

    useKeyboard(
        {
            onEnterCapture: () => {
                emitChange(value);
            },
            onEsc: () => onEditingChange?.(false),
            onTab: (event) => event.preventDefault(),
        },
        display,
        [emitChange, value, onEditingChange]
    );

    const handleContentEditableRef = useCallback(
        (instance: HTMLElement) => {
            editingRef?.(instance);
            contentEditableRef.current = instance;
        },
        [editingRef]
    );

    return (
        <EditingArea name={name} display={display}>
            <BubbleInput
                display={display}
                text={value}
                onTextChange={setValue}
                placeholder={label}
                hasClear={hasClear}
                innerRef={(instance) => (textinputRef.current = instance)}
                contentEditableRef={handleContentEditableRef}
                tabIndex={tabIndex}
                autoFocus={autoFocus}
                isMultiline={isMultiline}
            />
        </EditingArea>
    );
});
