import React, { useCallback, ComponentType } from "react";
import { FocusNode, FocusNodeRenderProps } from "../../../FocusScope";
import {
    startEditingEvent,
    stopEditingEvent,
    changeEvent,
} from "../../../events";
import { EntireProps } from "../../types/EntireProps";
import { ReadingProps } from "../../types/ReadingProps";
import { EditingProps } from "../../types/EditingProps";

export const EntireAttribute = <
    EnP extends EntireProps<RP, EdP>,
    RP extends ReadingProps<RV>,
    EdP extends EditingProps<EV>,
    RV = unknown,
    EV = unknown
>(
    props: EnP & {
        reading: ComponentType<RP>;
        editing?: ComponentType<EdP>;
    }
) => {
    const {
        isEditing,
        reading: Reading,
        editing: Editing,
        onEvent,
        onChange,
        onEditingChange,
        ...rest
    } = props;

    const handleEditingChange = useCallback(
        (isEditing: boolean) => {
            onEditingChange?.(isEditing);
            if (isEditing) {
                onEvent?.(startEditingEvent());
            } else {
                onEvent?.(stopEditingEvent());
            }
        },
        [onEditingChange, onEvent]
    );

    const handleChange = useCallback(
        (newValue: EV) => {
            onEvent?.(changeEvent());
            return onChange?.(newValue);
        },
        [onChange, onEvent]
    );

    const render = useCallback(
        ({ tabIndex, innerRef }: FocusNodeRenderProps) => {
            const readingProps: RP = (!isEditing || !Editing
                ? { ...rest, onEvent, tabIndex, readingRef: innerRef }
                : rest) as unknown as RP;
            const editingProps: EdP = (isEditing
                ? { ...rest, onEvent, tabIndex, editingRef: innerRef }
                : rest) as unknown as EdP;

            return (
                <>
                    <Reading
                        {...readingProps}
                        onEditingChange={handleEditingChange}
                        display={!isEditing || !Editing}
                    />
                    {Editing && (
                        <Editing
                            {...editingProps}
                            onEditingChange={handleEditingChange}
                            onChange={handleChange}
                            display={isEditing}
                        />
                    )}
                </>
            );
        },
        [isEditing, rest]
    );

    return <FocusNode render={render} isDisabled={props.isDisabled} />;
};
