import React, { useRef, useCallback, MouseEvent } from "react";
import { Text } from "../../../lego2/Text";
import { useFocus } from "../../../useFocus";
import { useKeyboard } from "../../../useKeyboard";
import { ReadingArea } from "../ReadingArea";
import { Label } from "../Label";
import { Value } from "../Value";
import { KeyValueReadingProps } from "./KeyValueReading.types";
import css from "./KeyValueReading.module.css";

export const KeyValueReading = <T extends unknown>(
    props: KeyValueReadingProps<T>
) => {
    const {
        label,
        name,
        readingValue,
        tabIndex,
        readingRef,
        display,
        onEditingChange,
        discardReadingArea = false,
        isReadingLoading = false,
        isEmpty = readingValue == null,
        render: outerRender,
        isDisabled,
    } = props;

    const essentialClickTimestamp = useRef(0);
    const readingAreaRef = useRef<HTMLDivElement>();
    const isFocused = useFocus(readingAreaRef);

    const handleEssentialClick = (event: MouseEvent) => {
        essentialClickTimestamp.current = event.timeStamp;
    };

    const render = useCallback(
        () =>
            outerRender ? (
                outerRender({
                    onEssentialClick: handleEssentialClick,
                })
            ) : (
                <>
                    <Label>{label}</Label>
                    <Value
                        className={css.KeyValueReading__value}
                        isEmpty={isEmpty}
                        isLoading={isReadingLoading}
                    >
                        <Text typography="body-short-m">
                            {readingValue as string}
                        </Text>
                    </Value>
                </>
            ),
        [label, readingValue, outerRender, isEmpty, isReadingLoading]
    );

    const emitEditingStart = useCallback(() => {
        if (isDisabled || isReadingLoading) {
            return;
        }

        onEditingChange?.(true);
    }, [onEditingChange, isDisabled, isReadingLoading]);

    useKeyboard(
        {
            onEnterCapture: (event) => {
                event.preventDefault();
                emitEditingStart();
            },
        },
        isFocused && display,
        [emitEditingStart]
    );

    const handleInnerRef = useCallback(
        (instance: HTMLDivElement) => {
            readingRef?.(instance);
            readingAreaRef.current = instance;
        },
        [readingRef]
    );

    const handleClick = useCallback(
        (event: MouseEvent) => {
            if (essentialClickTimestamp.current !== event.timeStamp) {
                emitEditingStart();
            } else {
                readingAreaRef.current?.focus();
            }
        },
        [emitEditingStart]
    );

    if (discardReadingArea) {
        return render();
    }

    return (
        <ReadingArea
            data-testid={`attribute_${name}`}
            tabIndex={tabIndex}
            innerRef={handleInnerRef}
            display={display}
            isHighlighted={!isDisabled && isFocused}
            onClick={handleClick}
            isHoverDisabled={isDisabled || isReadingLoading}
        >
            {render()}
        </ReadingArea>
    );
};
