import React, { FC, useState, useRef, useCallback } from "react";
import cx from "classnames";
import { useNullTextSelectionClick } from "../../../useNullTextSelectionClick";
import { ReadingAreaProps } from "./ReadingArea.types";
import { GridTemplate } from "../GridTemplate";
import css from "./ReadingArea.module.css";

export const ReadingArea: FC<ReadingAreaProps> = (props) => {
    const {
        innerRef,
        isHighlighted,
        tabIndex,
        children,
        display,
        isHoverDisabled,
        onClick,
        ...rest
    } = props;

    const rootRef = useRef<HTMLDivElement>();

    const [isHovered, setHovered] = useState(false);
    const className = cx(css.ReadingArea, {
        [css.ReadingArea_disabledHover]: isHoverDisabled,
        [css.ReadingArea_highlighted]: isHighlighted || isHovered,
    });

    const { onMouseEnter, onMouseLeave, ...nullTextSelectionHandlers } =
        useNullTextSelectionClick(
            rootRef,
            useCallback(
                (event) => {
                    onClick?.(event);
                },
                [onClick]
            )
        );

    const handleMouseEnter = useCallback(() => {
        onMouseEnter();
        if (isHoverDisabled) {
            return;
        }

        setHovered(true);
    }, [onMouseEnter, isHoverDisabled]);

    const handleMouseLeave = useCallback(() => {
        onMouseLeave();
        if (isHoverDisabled) {
            return;
        }

        setHovered(false);
    }, [onMouseLeave, isHoverDisabled]);

    const handleInnerRef = useCallback(
        (instance: HTMLDivElement) => {
            rootRef.current = instance;
            innerRef?.(instance);
        },
        [innerRef]
    );

    return (
        <GridTemplate
            {...rest}
            aria-label="reading area"
            display={display}
            innerRef={handleInnerRef}
            className={className}
            tabIndex={tabIndex}
            onMouseEnter={handleMouseEnter}
            onMouseLeave={handleMouseLeave}
            {...nullTextSelectionHandlers}
        >
            {children}
        </GridTemplate>
    );
};
