import React, { ComponentType, FC, memo, useEffect, useRef } from "react";
import { Grid } from "../Attribute2";
import {
    attributeKeyboardEvent,
    changeFailEvent,
    changeSuccessEvent,
} from "../events";
import { Textinput } from "./Textinput";
import { Select } from "./Select";
import {
    AttributesBySchemeProps,
    AttributesSchemeRow,
    AttributeComponentProps,
} from "./AttributesByScheme.types";

const defaultComponents: Record<
    string,
    ComponentType<AttributeComponentProps>
> = {
    TextInput: Textinput,
    Select,
};

export const AttributesByScheme: FC<AttributesBySchemeProps> = memo((props) => {
    const { scheme, components, getCommon, onCommonChange, onEvent } = props;

    const sentKeyboardNavigationLogRef = useRef(false);
    const gridRef = useRef<HTMLElement | null>();

    useEffect(() => {
        const gridNode = gridRef.current;
        if (!gridNode) {
            return;
        }

        const handleKeyDown = (event: KeyboardEvent) => {
            if (sentKeyboardNavigationLogRef.current) {
                return;
            }

            if (event.key === "Tab") {
                onEvent?.(null, attributeKeyboardEvent());
                sentKeyboardNavigationLogRef.current = true;
            }
        };

        gridNode.addEventListener("keydown", handleKeyDown);
        return () => {
            gridNode.removeEventListener("keydown", handleKeyDown);
        };

        // eslint-disable-next-line react-hooks/exhaustive-deps
    }, []);

    const handleChange = async (attribute: AttributesSchemeRow, value) => {
        const changeHandler =
            props.components?.[attribute.component]?.changeHandler ||
            onCommonChange;

        try {
            // eslint-disable-next-line
            await (changeHandler as any)?.(attribute, value);
            onEvent?.(attribute, changeSuccessEvent());
        } catch {
            onEvent?.(attribute, changeFailEvent());
        }
    };

    return (
        <Grid innerRef={(instance) => (gridRef.current = instance)}>
            {scheme.map((attribute) => {
                const AttributeComponent =
                    components?.[attribute.component]?.render ||
                    defaultComponents[attribute.component];
                if (!AttributeComponent) {
                    return null;
                }

                return (
                    <AttributeComponent
                        key={attribute.name}
                        // eslint-disable-next-line @typescript-eslint/no-explicit-any
                        attribute={attribute as any}
                        onChange={handleChange}
                        onEvent={onEvent}
                        {...getCommon?.(attribute)}
                        {...props.components?.[attribute.component]?.getProps?.(
                            attribute
                        )}
                    />
                );
            })}
        </Grid>
    );
});
