import React, { useState, useCallback, useMemo, FC } from "react";
import {
    Select as SelectAttribute,
    SelectEditingValue,
} from "../../Attribute2";
import { Event } from "../../types/Event";
import { stopEditingEvent } from "../../events";
import { idNameToIdText } from "../idNameToIdText";
import { AttributeComponentProps } from "../AttributesByScheme.types";
import { IdName } from "./Select.types";

export const Select: FC<
    AttributeComponentProps & {
        loadItems?: () => IdName[] | Promise<IdName[]>;
    }
> = (props) => {
    const { attribute, onChange, isDisabled, onEvent, loadItems } = props;
    const rawValue =
        (props.value as (IdName & { resolution?: IdName }) | null) || null;
    const [isReadingLoading, setReadingLoading] = useState(false);
    const [isEditing, setEditing] = useState(false);

    const value = useMemo(
        () => rawValue && idNameToIdText(rawValue),
        [rawValue]
    );

    const emitEvent = useCallback(
        (event: Event) => {
            onEvent?.(attribute, event);
        },
        [onEvent]
    );

    const handleChange = useCallback(
        async (newValue: SelectEditingValue) => {
            setReadingLoading(true);
            setEditing(false);
            emitEvent(stopEditingEvent());
            try {
                await onChange(attribute, newValue);
            } finally {
                setReadingLoading(false);
            }
        },
        [onChange, emitEvent]
    );

    let readingValue = value && value.text;
    if (rawValue && rawValue.resolution && rawValue.resolution.name) {
        readingValue = `${readingValue} (${rawValue.resolution.name})`;
    }

    return (
        <SelectAttribute
            isEditing={isEditing}
            onEditingChange={setEditing}
            isReadingLoading={isReadingLoading}
            label={attribute.label}
            name={attribute.name}
            readingValue={readingValue}
            editingValue={value}
            onChange={handleChange}
            isDisabled={isDisabled}
            onEvent={emitEvent}
            onLoad={async () => {
                const items = (await loadItems?.()) || [];
                if (Array.isArray(items)) {
                    return items.map(idNameToIdText);
                }

                return [];
            }}
        />
    );
};
