import React, { CSSProperties, FC, useRef } from "react";
import colorFn from "color";
import cx from "classnames";
import Button from "../lego2/Button";
import Icon from "../lego2/Icon";
import { useFocus } from "../useFocus";
import css from "./Bubble.module.css";
import { BubbleProps } from "./Bubble.types";
import { processColor } from "./processColor";

export const Bubble: FC<BubbleProps> = (props) => {
    const {
        view = "default",
        color: outerColor,
        onClick,
        canDelete = false,
        onDelete,
        children,
        render,
        title,
        cursor,
        innerRef,
        enableFocusStyles = true,
    } = props;

    const formattedColor = processColor(outerColor);

    const rootRef = useRef<HTMLSpanElement>();
    const isFocused = useFocus(rootRef);

    const className = cx(props.className, css.Bubble, {
        [css.Bubble_focused]: isFocused && enableFocusStyles,
    });
    const color =
        isFocused && enableFocusStyles
            ? colorFn(formattedColor).lighten(0.1).rgb().string()
            : formattedColor;
    const style: CSSProperties = {
        cursor,
    };

    if (view !== "clear") {
        style.backgroundColor = color;
    }

    let lastDeleteTimeStamp = -1;

    const handleDelete = (event) => {
        if (!onDelete) return;
        lastDeleteTimeStamp = event.timeStamp;
        onDelete(event);
    };

    const handleClick = (event) => {
        if (!onClick || lastDeleteTimeStamp === event.timeStamp) return;
        onClick(event);
    };

    const handleRef = (instance: HTMLSpanElement | null) => {
        rootRef.current = instance!;
        innerRef?.(instance!);
    };

    return (
        <span
            className={className}
            style={style}
            onClick={handleClick}
            role={onClick && "button"}
            title={title}
            tabIndex={-1}
            ref={handleRef}
            data-bubble-root
        >
            {render ? (
                render(css.Bubble__text)
            ) : (
                <span className={css.Bubble__text}>{children}</span>
            )}
            {canDelete && (
                <Button
                    size="xs"
                    view="clear"
                    className={css.Bubble__remove}
                    onClick={handleDelete}
                    tabIndex={-1}
                    icon={(cn) => (
                        <Icon size="xs" svg="close2" className={cn} />
                    )}
                    aria-label="remove"
                />
            )}
        </span>
    );
};
