import React, { Component } from 'react';
import cx from 'classnames';
import { ContentEditableProps } from './ContentEditable.types';
import css from './ContentEditable.module.css';

export class ContentEditable extends Component<ContentEditableProps> {
  private element: HTMLDivElement;

  private defaultValue: string = '';

  public constructor(props) {
    super(props);
    this.defaultValue = props.value;
    this.handleRef = this.handleRef.bind(this);
    this.handleInput = this.handleInput.bind(this);
    this.handlePaste = this.handlePaste.bind(this);
  }

  private handlePaste(event: ClipboardEvent) {
    event.preventDefault();
    const text = event.clipboardData!.getData('text/plain');
    document.execCommand('insertText', false, text);
  }

  public componentDidMount() {
    this.autoFocusSelectionIfNeeded();
    this.element.addEventListener('paste', this.handlePaste);
  }

  public componentWillUnmount() {
    this.element.removeEventListener('paste', this.handlePaste);
  }

  public shouldComponentUpdate(nextProps: ContentEditableProps) {
    return (
      (this.props.value !== nextProps.value && nextProps.value !== this.element.textContent) ||
      this.props.isDisabled !== nextProps.isDisabled ||
      this.props.tabIndex !== nextProps.tabIndex ||
      this.props.display !== nextProps.display ||
      (this.props.placeholder !== nextProps.placeholder && !this.props.value)
    );
  }

  private autoFocusSelectionIfNeeded() {
    const root = this.element;
    const { display, autoFocus } = this.props;

    if (!display) {
      return;
    }

    if (autoFocus) {
      requestAnimationFrame(() => {
        // фокусим в следующем фрейме, на случай,
        // если в этом фрейме будет сфокусирован другой элемент.
        // так мы отберем у него фокус в любом случае.
        root.focus();
        const textNode = root.childNodes[0];
        const range = document.createRange();
        if (textNode) {
          range.selectNodeContents(textNode);
        } else {
          range.selectNodeContents(root);
        }
        const selection = window.getSelection()!;
        selection.removeAllRanges();
        selection.addRange(range);
      });
    }
  }

  public componentDidUpdate(prevProps: ContentEditableProps) {
    const { display } = this.props;

    if (prevProps.display !== display) {
      this.autoFocusSelectionIfNeeded();
    }

    if (this.props.value !== this.element.textContent) {
      this.element.textContent = this.props.value || '';
    }
  }

  private handleRef(instance: HTMLDivElement) {
    this.element = instance!;
    this.props.innerRef?.(instance!);
  }

  private handleInput() {
    const { value, onChange } = this.props;
    const root = this.element;
    let newValue = root.textContent || '';
    if (newValue === '\n') {
      newValue = '';
    }

    if (newValue !== value) {
      onChange?.(newValue);
    }

    if (!newValue && this.element.children[0] && this.element.children[0].tagName === 'BR') {
      this.element.children[0].remove();
    }
  }

  public render() {
    const { props } = this;
    const { isDisabled = false, isMultiline = false } = props;

    const className = cx(props.className, css.ContentEditable, {
      [css.ContentEditable_disabled]: isDisabled,
      [css.ContentEditable_singleline]: !isMultiline,
    });

    return (
      <div
        className={className}
        suppressContentEditableWarning
        contentEditable={!props.isDisabled}
        tabIndex={!props.isDisabled ? props.tabIndex : undefined}
        role="textbox"
        placeholder={props.placeholder}
        onKeyDown={props.onKeyDown}
        onInput={this.handleInput}
        ref={this.handleRef}
        children={this.defaultValue}
      />
    );
  }
}
