import { FC, useRef, useEffect, cloneElement, useCallback } from "react";
import { observer } from "mobx-react-lite";
import { useFocus } from "../../useFocus";
import { useFocusScope } from "../FocusScopeContext";
import { FocusNodeProps } from "./FocusNode.types";

export const FocusNode: FC<FocusNodeProps> = observer((props) => {
    const { children, render, isDisabled } = props;
    const service = useFocusScope();

    const registeredIndex = useRef<number | null>(null);
    const rootRef = useRef<HTMLElement | null>();
    const isFocused = useFocus(rootRef);

    const handleRef = useCallback((instance: HTMLElement | null) => {
        if (registeredIndex.current != null && service) {
            service.replaceNode(registeredIndex.current, instance);
        }
        rootRef.current = instance;
    }, []);

    useEffect(() => {
        const rootNode = rootRef.current;
        if (!rootNode || !service) {
            return;
        }

        registeredIndex.current = service.registerNode(rootNode);
        return () => {
            service.unregisterNode(registeredIndex.current!);
        };
    }, []);

    useEffect(() => {
        if (!service || !isFocused || registeredIndex.current == null) {
            return;
        }

        service.setLastFocusedIndex(registeredIndex.current);
    }, [isFocused]);

    if (!service) {
        console.error("FocusNode: null focus scope.");
        return null;
    }

    const propsToRender: {
        innerRef: (instance: HTMLElement) => void;
        ref: (instance: HTMLElement) => void;
        tabIndex: number;
    } = {
        innerRef: () => {},
        ref: () => {},
        tabIndex: isDisabled ? -1 : service.nodeTabIndex,
    };

    if (render) {
        propsToRender.innerRef = handleRef;
        return render(propsToRender);
    }

    if (children) {
        propsToRender.ref = handleRef;
        return cloneElement(children, propsToRender);
    }

    return null;
});
