import { makeObservable, action, observable, computed } from 'mobx';

export type LeaveSource = 'mouse' | 'keyboard';

export class FocusScopeService {
  private parentScope: FocusScopeService | null;

  public constructor(parentScope?: FocusScopeService | null) {
    if (parentScope) {
      this.parentScope = parentScope;
    }

    makeObservable<FocusScopeService, '_isWithin'>(this, {
      _isWithin: observable,
      scopeTabIndex: computed,
      nodeTabIndex: computed,
      enter: action.bound,
      leave: action.bound,
      setLastFocusedIndex: action.bound,
    });
  }

  private lastFocusedIndex: number = -1;

  private lastLeaveSource: LeaveSource = 'mouse';

  private array: (HTMLElement | null)[] = [];

  private scopeNode: HTMLElement;

  private _isWithin: boolean = false;

  public registerScopeNode(node: HTMLElement) {
    this.scopeNode = node;
    this.scopeNode.tabIndex = this.scopeTabIndex;
    this.scopeNode.dataset.focusScope = 'true';
  }

  public closestScopeToNode(node: HTMLElement): HTMLElement | null {
    const closestScope = node.closest('[data-focus-scope]') as HTMLElement;
    if (closestScope === node) {
      const parentNode = node.parentElement;
      if (!parentNode) {
        return null;
      }

      return parentNode.closest('[data-focus-scope]');
    }

    return closestScope;
  }

  public registerNode(node: HTMLElement): number {
    const index = this.array.push(node) - 1;
    node.tabIndex = this.nodeTabIndex;
    return index;
  }

  public unregisterNode(index: number) {
    this.array[index] = null;
  }

  public replaceNode(index: number, node: HTMLElement | null) {
    this.array[index] = node;
    if (index === this.lastFocusedIndex && this.lastLeaveSource === 'keyboard') {
      node?.focus({
        preventScroll: true,
      });
    }
  }

  public get scopeTabIndex(): number {
    return !this.isWithinParentScope() || this.isWithin() ? -1 : 0;
  }

  public get nodeTabIndex(): number {
    return this.isWithin() ? 0 : -1;
  }

  public enter() {
    if (this._isWithin) {
      return;
    }

    this._isWithin = true;
  }

  public focusRelevantNode() {
    if (this.lastFocusedIndex !== -1) {
      const node = this.array[this.lastFocusedIndex];
      node?.focus();
    } else {
      const firstNode = this.array[0];
      firstNode?.focus();
    }
  }

  public leave(source?: LeaveSource) {
    if (!this._isWithin) {
      return;
    }

    this.lastLeaveSource = source || 'mouse';
    this._isWithin = false;
  }

  public focusScopeNode() {
    this.scopeNode.focus();
  }

  private isWithinParentScope() {
    return this.parentScope ? this.parentScope.isWithin() : true;
  }

  public isWithin() {
    return this._isWithin;
  }

  public setLastFocusedIndex(index: number) {
    this.lastFocusedIndex = index;
  }
}
