import React from "react";
import cx from "classnames";
import { Resizable } from "../Resizable";
import { Pin } from "./Pin";
import PanelButton from "./PanelButton";
import CloseButton from "./CloseButton";
import { Header } from "./Header";
import { Position, Tone, PanelScheme, PanelProps, PanelId } from "./types";
import css from "./Panel.module.css";

const ResizeDirection = {
    [Position.Right]: "left",
    [Position.Left]: "right",
};

class Panel extends React.Component<PanelProps> {
    public static defaultProps = {
        hasToggleButton: true,
        hasPinButton: true,
        canResize: true,
        position: Position.Right,
        tone: Tone.White,
        minWidth: 200,
        maxWidth: 400,
        isNewDesign: false,
        defaultSize: {
            width: 320,
        },
    };

    private ref = React.createRef<HTMLDivElement>();
    public static Position = Position;

    private cacheKeepMountPanels = new Map<PanelId, PanelScheme>();

    private setPanelId = (id: PanelScheme["id"]) => () => {
        const { setPanelId, panelId } = this.props;
        const payload = id !== panelId ? id : 0;
        setPanelId(payload);
    };

    private renderPanel = (
        panel: PanelScheme,
        currentPanel: PanelScheme | null
    ) => {
        const { isNewDesign } = this.props;

        const contentSize = isNewDesign ? "m" : "xs";
        return (
            <div
                key={panel.id}
                className={cx(
                    css.Panel__content,
                    css[`Panel__content_${contentSize}`],
                    this.props.contentClassName,
                    {
                        [css.Panel__content_padding]: panel.padding ?? true,
                        [css.Panel__content_overflow]: panel.overflow ?? true,
                        [css.Panel__content_hidden]: !(
                            currentPanel && panel.id === currentPanel.id
                        ),
                    }
                )}
            >
                {panel.content}
            </div>
        );
    };

    private renderContent = (panel: PanelScheme | null) => {
        const { panels } = this.props;

        panels.forEach((p) => {
            if (p.mountMode === "always") {
                this.cacheKeepMountPanels.set(p.id, p);
            }
        });

        const renderedPanels: React.ReactNode[] = [];

        if (panel) {
            if (panel.keepMount || panel.mountMode === "lazy") {
                this.cacheKeepMountPanels.set(panel.id, panel);
            } else if (panel.mountMode !== "always") {
                renderedPanels.push(this.renderPanel(panel, panel));
            }
        }

        this.cacheKeepMountPanels.forEach((value) => {
            renderedPanels.push(this.renderPanel(value, panel));
        });

        return renderedPanels;
    };

    private getPanelById = (id?: PanelId) => {
        const { panels } = this.props;
        return (id && panels.find((p) => p.id === id)) || null;
    };

    private panelStateChange = () => {
        const { panelId, name, onPanelStateChange } = this.props;
        const panel = this.getPanelById(panelId);
        onPanelStateChange?.(
            {
                details: {
                    visibilityState: panel ? "visible" : "hidden",
                    panelName: name,
                    panelId: panel?.id,
                    panelTitle: panel?.title,
                },
            },
            this.ref.current
        );
    };

    componentDidMount() {
        this.panelStateChange();
    }

    componentDidUpdate(prevProps) {
        if (prevProps.panelId !== this.props.panelId) {
            this.panelStateChange();
        }
    }

    public render() {
        const {
            pin,
            name,
            className,
            wrapClassName,
            position,
            tone,
            hasToggleButton,
            hasPinButton,
            canResize,
            defaultSize,
            width,
            minWidth,
            maxWidth,
            panels,
            panelId,
            hidden,
            renderPopupActions,
            onSizeChange,
            onPinClick,
            size,
            isNewDesign,
            htmlAttributes,
        } = this.props;

        const panel = this.getPanelById(panelId);
        const pinned = !hasPinButton || pin;
        const header = isNewDesign ? (
            <Header
                hasToggleButton={hasToggleButton}
                onClose={this.setPanelId(0)}
                hasPinButton={hasPinButton}
                panel={panel}
                name={name}
                position={position}
                renderPopupActions={renderPopupActions}
                pin={pinned}
            />
        ) : (
            <div className={css.Panel__header}>
                {hasToggleButton && (
                    <CloseButton onClose={this.setPanelId(0)} />
                )}
                {hasPinButton && (
                    <Pin name={name} pin={pinned} onClick={onPinClick} />
                )}
                {panel && (
                    <span className={css.Panel__title}>{panel.title}</span>
                )}
            </div>
        );
        const hideOpenedToggle = isNewDesign && panels.length === 1;
        return (
            <div
                {...htmlAttributes}
                className={cx(
                    wrapClassName,
                    css.Panel,
                    css[`Panel_position_${position}`],
                    css[`Panel_tone_${tone}`],
                    {
                        [css.Panel_pin]: pinned,
                        [css.Panel_open]: panel,
                        [css.Panel_hidden]: hidden,
                    }
                )}
            >
                {
                    <Resizable
                        enable={canResize ? ResizeDirection[position] : "none"}
                        wrap
                        name={name}
                        defaultSize={defaultSize!}
                        minWidth={minWidth}
                        maxWidth={maxWidth}
                        className={cx(
                            css.Panel__resizable,
                            css[`Panel__resizable_width_${width}`],
                            {
                                [css.Panel__resizable_pin]: pinned,
                                [css.Panel__resizable_hidden]: !panel,
                            }
                        )}
                        disabled={!(panel && canResize)}
                        onSizeChange={onSizeChange}
                        size={size}
                    >
                        <div
                            className={cx(
                                className,
                                "resizable",
                                css.Panel__toggleWrap,
                                {
                                    [css.Panel__toggleWrap_unpin]: !pinned,
                                }
                            )}
                        >
                            <div ref={this.ref} className={css.Panel__panel}>
                                {header}
                                {this.renderContent(panel)}
                            </div>
                        </div>
                    </Resizable>
                }
                <div
                    className={cx(css.Toggle, {
                        [css.Toggle_hidden]: hideOpenedToggle && panel,
                    })}
                >
                    {panels.map((panel) => (
                        <PanelButton
                            key={panel.id}
                            panel={panel}
                            setPanelId={this.setPanelId}
                            currentPanelId={panelId}
                        />
                    ))}
                </div>
            </div>
        );
    }
}

export default Panel;
