import * as React from "react";
import debounce from "lodash/debounce";
import throttle from "lodash/throttle";
import ResizableComponent, { ResizableBase } from "./Resizable";

export type Size = { width?: number | string; height?: number | string };

interface Props {
    defaultSize: Size;
    size: Size;
    onSizeChange?: (size: Size) => void;
    name: string;
    enable: any;
    minWidth?: number;
    maxWidth?: number;
    className?: string;
    wrap?: boolean;
    disabled?: boolean;
}

export class Resizable extends React.Component<Props> {
    public static UPDATE_DELAY = 300;

    private nodeRef = React.createRef<ResizableBase>();

    private updateSize = debounce(() => {
        if (this.nodeRef.current) {
            this.props.onSizeChange?.({
                width: this.nodeRef.current.resizable.getBoundingClientRect()
                    .width,
            });
        }
    }, Resizable.UPDATE_DELAY);

    private resizeNotify = throttle(() => {
        window.dispatchEvent(new Event("resize"));
    }, Resizable.UPDATE_DELAY);

    public componentDidMount() {
        if (!this.props.disabled) {
            this.mount();
        }
    }

    public componentDidUpdate(prevProps): void {
        if (this.props.disabled && !prevProps.disabled) {
            this.unmount();
        }

        if (!this.props.disabled && prevProps.disabled) {
            this.mount();
        }
    }

    public componentWillUnmount() {
        this.unmount();
    }

    private handleResizeStop = (e, direction, ref, d) => {
        const newSize: { width?: number; height?: number } = {};

        const newWidth = this.props.size.width + d.width;
        if (typeof newWidth === "number" && !Number.isNaN(newWidth)) {
            newSize.width = newWidth;
        }

        const newHeight = this.props.size.height + d.height;
        if (typeof newHeight === "number" && !Number.isNaN(newHeight)) {
            newSize.height = newHeight;
        }
        this.props.onSizeChange?.(newSize);

        this.resizeNotify();

        this.resizeListenerOn();
    };

    private resizeListenerOn = () => {
        window.addEventListener("resize", this.updateSize);
    };

    private resizeListenerOff = () => {
        window.removeEventListener("resize", this.updateSize);
    };

    private handleResizeStart = (e) => {
        e.preventDefault();

        this.resizeListenerOff();
    };

    private mount() {
        const { defaultSize, size, onSizeChange } = this.props;

        this.resizeListenerOn();
        if (!size && defaultSize) {
            onSizeChange?.(defaultSize);
        }

        // update size if initial size more then max-width
        this.updateSize();
    }

    private unmount() {
        this.resizeListenerOff();
        this.updateSize.cancel();
        this.resizeNotify.cancel();
    }

    public render() {
        const { name, onSizeChange, defaultSize, ...reset } = this.props;

        return (
            <ResizableComponent
                onResizeStart={this.handleResizeStart}
                onResize={this.resizeNotify}
                onResizeStop={this.handleResizeStop}
                {...reset}
                ref={this.nodeRef}
            />
        );
    }
}
