import React, { useRef, useEffect, useCallback, useState } from "react";
import throttle from "lodash/throttle";
import cx from "classnames";
import css from "./Scrollable.module.css";
import { ScrollableProps, ScrollDirection } from "./Scrollable.types";
import { getScrollDirection } from "./Scrollable.utils";
import { SCROLL_THROTTLE_WAIT } from "./Scrollable.config";

export const Scrollable: React.FC<ScrollableProps> = (props) => {
    const { className, scrollClassName, withModalPadding, children } = props;

    const ref = useRef<HTMLDivElement>(null);
    const [scrollDirection, setScrollDirection] =
        useState<ScrollDirection>("none");

    const handleScroll = useCallback(
        throttle(() => {
            if (ref.current) {
                setScrollDirection(getScrollDirection(ref.current));
            }
        }, SCROLL_THROTTLE_WAIT),
        []
    );

    useEffect(() => {
        handleScroll();

        const resizeObserver = new ResizeObserver(handleScroll);
        resizeObserver.observe(ref.current!);

        return () => {
            resizeObserver.unobserve(ref.current!);
            handleScroll.cancel();
        };
    }, []);

    const scrollerCn = cx(css.Scrollable, className, {
        [css.Scrollable_top]:
            scrollDirection === "top" || scrollDirection === "both",
        [css.Scrollable_bottom]:
            scrollDirection === "bottom" || scrollDirection === "both",
        [css.Scrollable_modalPadding]: withModalPadding,
    });
    const scrollerScrollCn = cx(css.Scrollable__scroll, scrollClassName);

    return (
        <div className={scrollerCn}>
            <div onScroll={handleScroll} ref={ref} className={scrollerScrollCn}>
                {children}
            </div>
        </div>
    );
};
