import React, { forwardRef } from "react";
import cx from "classnames";
import { compose } from "@bem-react/core";
import {
    Button,
    IButtonProps,
} from "@yandex-lego/components/Button/desktop/bundle";
import { LegoSizeProp } from "../types";
import { withSizeXS } from "./_size/Button_size_xs";
import { withViewDanger } from "./_view/Button_view_danger";
import { withViewSuccess } from "./_view/Button_view_success";
import "./Button.css";

export enum Direction {
    Horizontal = "horizontal",
    Vertical = "vertical",
}

export type ButtonPinProp =
    | "brick-brick"
    | "brick-circle"
    | "brick-clear"
    | "brick-round"
    | "circle-brick"
    | "circle-circle"
    | "circle-clear"
    | "clear-brick"
    | "clear-circle"
    | "clear-clear"
    | "clear-round"
    | "round-brick"
    | "round-clear";

export type ButtonViewProp =
    | "default"
    | "action"
    | "clear"
    | "link"
    | "pseudo"
    | "raised"
    | "danger"
    | "success";

export interface ButtonProps extends Omit<IButtonProps, "size" | "view"> {
    size?: LegoSizeProp;
    view?: ButtonViewProp;
    style?: React.CSSProperties;
    title?: string;
    direction?: Direction;
    className?: string;
    tabIndex?: number;
    target?: "_blank";
    onMouseEnter?: (event: React.MouseEvent) => void;
    onMouseLeave?: (event: React.MouseEvent) => void;
    /**
     * for type = "link"
     * */
    url?: string;
}

const ButtonWithCustomMod = compose(
    withSizeXS,
    withViewDanger,
    withViewSuccess
)(Button) as React.FC<ButtonProps>;

const ButtonExtend: React.FC<ButtonProps> = (props) => {
    const {
        direction,
        className,
        // https://st.yandex-team.ru/CRM-15295
        tooltipCloseDelay,
        stopPropagation,
        issueId,
        oId,
        redirect,
        fieldProps,
        alert,
        isLoading,
        onClear,
        previewComponent,
        dispatch,
        onCreate,
        onFocusChange,
        focused,
        hasClear,
        extraProps,
        asButton,
        ...other
        // eslint-disable-next-line
    } = props as any;

    return (
        <ButtonWithCustomMod
            data-testid="Button"
            className={cx(className, `Button2_direction_${direction}`)}
            {...other}
        />
    );
};

type RefForwardedButton = {
    Direction: typeof Direction;
} & React.ForwardRefExoticComponent<
    ButtonProps & React.RefAttributes<HTMLElement>
>;

const RefForwardedButton: RefForwardedButton = forwardRef(
    (props: ButtonProps, ref: React.RefObject<HTMLElement>) => {
        const { innerRef, ...restProps } = props;

        return <ButtonExtend {...restProps} innerRef={innerRef || ref} />;
    }
) as RefForwardedButton;

RefForwardedButton.defaultProps = {
    size: "s",
    view: "default",
    direction: Direction.Horizontal,
};

RefForwardedButton.Direction = Direction;

export default RefForwardedButton;
