import React from "react";
import cx from "classnames";
import Button from "../Button";
import Icon from "../Icon";
import { Title } from "../../Title";
import { Scrollable } from "../../Scrollable";
import { ModalContentProps } from "./ModalContent.types";

const crossIconProvider = (iconCls) => <Icon className={iconCls} svg="cross" />;
const minimizeIconProvider = (iconMnmz) => (
    <Icon className={iconMnmz} svg="minimize" />
);

const ModalContent: React.FC<ModalContentProps> = (props) => {
    const {
        children,
        onClose,
        onMinimize,
        title,
        paddingTop,
        className: outerClassName,
        footerComponent,
        footerLeftChildren,
        footerRightChildren,
        scrollableContent,
    } = props;

    let className = cx("ModalContent", outerClassName, {
        ModalContent_hasClose: onClose,
        ModalContent_hasMinimize: onMinimize,
    });

    if (paddingTop) {
        className = cx(className, "ModalContent_paddingTop");
    }

    const renderedClose = onClose && (
        <Button
            className="ModalContent-Close"
            onClick={onClose}
            view="clear"
            size="m"
            icon={crossIconProvider}
        />
    );

    const renderedMinimize = onMinimize && (
        <Button
            className="ModalContent-Minimize"
            onClick={onMinimize}
            view="clear"
            size="m"
            icon={minimizeIconProvider}
        />
    );

    const renderLeftChildren = footerLeftChildren && (
        <div className="ModalContent__footer_left">{footerLeftChildren}</div>
    );

    const renderRightChildren = footerRightChildren && (
        <div className="ModalContent__footer_right">{footerRightChildren}</div>
    );

    const renderTitle = title && (
        <Title className="ModalContent__title">{title}</Title>
    );

    const renderContent = scrollableContent ? (
        <Scrollable withModalPadding>{children}</Scrollable>
    ) : (
        children
    );

    return (
        <div className={className}>
            {renderedClose}
            {renderedMinimize}
            {renderTitle}
            {renderContent}
            {footerComponent
                ? footerComponent(
                      "ModalContent__footer",
                      renderRightChildren,
                      renderLeftChildren
                  )
                : (renderLeftChildren || renderRightChildren) && (
                      <div className="ModalContent__footer">
                          {renderLeftChildren}
                          {renderRightChildren}
                      </div>
                  )}
        </div>
    );
};

export default ModalContent;
