import * as React from "react";
import { withProps } from "recompose";
import LegoModal, { ModalProps as LegoModalProps } from "./Modal";

type ModalProps<TContentProps> = {
    contentProps?: TContentProps;
} & LegoModalProps;

type ModalStateProps<TContentProps> = {
    contentProps?: TContentProps;
} & LegoModalProps;
interface ModalState<TContentProps> {
    props: ModalStateProps<TContentProps> | null;
    visible: boolean;
}

interface ModalLink<TContentProps> {
    setState: (state: ModalState<TContentProps>) => void;
}

interface CreateContextConfig<TContentProps> {
    modalProps?: LegoModalProps;
    contentProps?: TContentProps;
}

export default <TContentProps extends {} = {}>(
        config: CreateContextConfig<TContentProps> = {}
    ) =>
    (Content) => {
        let modalLink: ModalLink<TContentProps> | null = null;

        const open = (props: TContentProps) => {
            if (modalLink) {
                modalLink.setState({ props, visible: true });
            }
        };

        const defaultValue = { open };

        const Context = React.createContext(defaultValue);

        const Provider = ({ children }) => (
            <Context.Provider value={defaultValue}>{children}</Context.Provider>
        );

        class Modal extends React.Component<
            ModalProps<TContentProps>,
            ModalState<TContentProps>
        > {
            public constructor(props) {
                super(props);
                this.state = {
                    props: {},
                    visible: false,
                };
            }

            public componentDidMount() {
                modalLink = this;
            }

            public componentWillUnmount() {
                modalLink = null;
            }

            private close = () => {
                this.setState({ props: null, visible: false });
            };

            public render() {
                return (
                    <LegoModal
                        visible={this.state.visible}
                        {...this.props}
                        onClose={this.close}
                        {...config.modalProps}
                    >
                        <Content
                            {...config.contentProps}
                            {...this.state.props}
                            {...this.props.contentProps}
                            close={this.close}
                        />
                    </LegoModal>
                );
            }
        }

        const withModalOpen = withProps((props) => ({
            open: () => open(props),
        }));

        const withOpen = withProps(() => ({
            open,
        }));

        return {
            Context,
            Provider,
            Modal,
            open,
            Consumer: Context.Consumer,
            withModalOpen,
            withOpen,
        } as any;
    };
