import * as React from 'react';
import Popup, { PopupProps } from './Popup';

/**
 * Обертка над Popup.
 * Пробрасывает свойства и методы элементам попапа для
 * императивного управления состоянием.
 */

// Передается элементам
type PopupBag<TExtraProps extends {} = {}> = {
  visible: boolean;
  show: () => void;
  hide: () => void;
  toggle: () => void;
} & TExtraProps;

export type ElementAsRenderProp<TExtraProps extends {} = {}> = (
  popupBag: PopupBag<TExtraProps>,
  props: ImperativePopupProps,
) => React.ReactNode;

interface ToggleCloneProps {
  onClick: () => void;
  // Для anchor @yandex-lego/components/Popup
  ref: React.Ref<HTMLElement>;
}

export interface ImperativePopupProps extends PopupProps {
  content?: ElementAsRenderProp;
  toggle?: ElementAsRenderProp<ToggleCloneProps>;
}

interface ImperativePopupState {
  visible: boolean;
}

class ImperativePopup extends React.Component<ImperativePopupProps, ImperativePopupState> {
  private popupBag: PopupBag;

  private toggleNode: React.RefObject<HTMLElement>;

  public constructor(props: ImperativePopupProps) {
    super(props);
    this.state = {
      visible: false,
    };

    this.popupBag = {
      visible: false,
      show: this.show,
      hide: this.hide,
      toggle: this.toggle,
    };
  }

  public componentDidUpdate(): void {
    this.popupBag.visible = this.state.visible;
  }

  private show = (): void => {
    this.setState({ visible: true });
  };

  private hide = (): void => {
    this.setState({ visible: false });
  };

  private toggle = (): void => {
    if (this.state.visible) {
      this.hide();
    } else {
      this.show();
    }
  };

  private getToggleRef = (node: HTMLElement): void => {
    this.toggleNode = {
      current: node,
    };
  };

  public render(): React.ReactNode {
    const { toggle, content, ...other } = this.props;

    /**
     * Пробрасываем onClick и ref через клон,
     * чтобы не делать это постоянно снаружи.
     * Но в toggle все равно пробрасываем, в случае,
     * если верхний элемент toggle будет React.Fragment.
     */
    const renderedToggle =
      toggle &&
      React.cloneElement<ToggleCloneProps>(
        toggle(
          {
            ...this.popupBag,
            onClick: this.toggle,
            ref: this.getToggleRef,
          },
          this.props,
        ) as React.ReactElement,
        {
          onClick: this.toggle,
          ref: this.getToggleRef,
        },
      );

    const renderedContent = content && content(this.popupBag, this.props);

    return (
      <React.Fragment>
        {renderedToggle}
        <Popup
          target="anchor"
          anchor={this.toggleNode}
          visible={this.state.visible}
          onOutsideClick={this.hide}
          {...other}
        >
          {renderedContent}
        </Popup>
      </React.Fragment>
    );
  }
}

export default ImperativePopup;
