import * as React from "react";
import { mergeAllRefs, mergeRefs } from "@yandex-lego/components/lib/mergeRefs";
import { compose as bemCompose } from "@bem-react/core";
import {
    Popup,
    IPopupProps,
} from "@yandex-lego/components/Popup/desktop/bundle";
import { Boundary } from "@yandex-lego/components/usePopper";
import { withDirections, Direction } from "../withOldDirections";
import { withProps, compose } from "recompose";
import cx from "classnames";
import { withZIndex } from "@yandex-lego/components/withZIndex";
import { withStopPropagation } from "./withStopPropagation";
import { withPaddingS, withPaddingXS } from "./_padding";
import { withViewError } from "./_view";
import "./Popup.css";

export type PopupProps = {
    padding?: "xs" | "s";
    target?: "anchor";
    anchor?: React.RefObject<HTMLElement>;
    childrenClassName?: string;
    view?: "default" | "error";
    theme?: "normal";
    mainOffset?: number;
    hasPadding?: boolean;
    secondaryOffset?: number;
    // eslint-disable-next-line
    unstable_essentialRefs?: React.RefObject<HTMLElement | undefined>[];
    boundary?: Boundary;
    /** @deprecated use direction */
    directions?: Direction[] | Direction;
    autoZIndex?: boolean;
    stopPropagation?: boolean;
    /** @deprecated use onClose */
    onOutsideClick?: (event: MouseEvent) => void;
    continuePropagation?: boolean;
    popupRef?: React.RefObject<HTMLDivElement> | null;
    parentProps?: PopupProps;
} & IPopupProps;

const PopupWithCustomMod = bemCompose(
    withPaddingS,
    withPaddingXS,
    withViewError,
    withDirections
)(Popup) as React.FC<PopupProps>;

export const PopupContext = React.createContext<PopupProps>({});

class PopupWrap extends React.Component<PopupProps> {
    private readonly innerRef: React.RefObject<HTMLInputElement>;
    private readonly popupRef: React.RefObject<HTMLInputElement>;

    constructor(props) {
        super(props);
        this.innerRef = React.createRef();
        this.popupRef = React.createRef();
    }

    private handlePopupClose: IPopupProps["onClose"] = (event, source) => {
        const { onClose, onOutsideClick, parentProps } = this.props;

        (onClose || onOutsideClick)?.(event, source);

        if (parentProps?.continuePropagation && parentProps.popupRef?.current) {
            const eventPath = event.composedPath();
            const isElementInPath = eventPath.includes(
                parentProps.popupRef.current
            );
            if (!isElementInPath) {
                (parentProps.onClose || parentProps.onOutsideClick)?.(
                    event,
                    source
                );
            }
        }
    };

    public render() {
        const {
            mainOffset = 7,
            children,
            childrenClassName: outerChildrenClassName,
            onOutsideClick,
            onClose,
            hasPadding = false,
            // https://st.yandex-team.ru/CRM-15295
            subModalsCount,
            onOpenSubModal,
            onCloseSubModal,
            ...restProps
            // eslint-disable-next-line
        } = this.props as any;
        const forwardProps = { ...restProps };
        const bodyRef: React.RefObject<HTMLElement> = {
            current: document.body,
        };

        const childrenClassName = cx(
            outerChildrenClassName,
            "Popup2-ChildrenWrap",
            {
                "Popup2-ChildrenWrap_hasPadding": hasPadding,
            }
        );

        forwardProps.children = (
            <div className={childrenClassName}>{children}</div>
        );

        forwardProps.onClose = this.props.parentProps?.continuePropagation
            ? this.handlePopupClose
            : onClose || onOutsideClick;

        return (
            <PopupContext.Provider
                value={{
                    ...forwardProps,
                    popupRef: mergeRefs(this.innerRef, this.popupRef),
                }}
            >
                <PopupWithCustomMod
                    data-testid="Popup"
                    view="default"
                    mainOffset={mainOffset}
                    boundary={bodyRef}
                    {...forwardProps}
                    innerRef={mergeAllRefs(this.innerRef, this.props.innerRef)}
                    target={forwardProps.anchor && forwardProps.target}
                />
            </PopupContext.Provider>
        );
    }
}

const RefForwardedPopup = React.forwardRef(
    (props: PopupProps, ref: React.RefObject<HTMLDivElement>) => {
        const { innerRef, ...restProps } = props;

        return (
            <PopupContext.Consumer>
                {(props) => (
                    <PopupWrap
                        {...restProps}
                        parentProps={props}
                        innerRef={innerRef || ref}
                    />
                )}
            </PopupContext.Consumer>
        );
    }
);

export default compose(
    withProps((props) => ({
        visible: props.visible,
    })),
    withStopPropagation,
    withZIndex
)(RefForwardedPopup) as React.ForwardRefExoticComponent<PopupProps>;
