import React, { Component, ReactNode, cloneElement, createRef } from 'react';
import { mergeAllRefs } from '@yandex-lego/components/lib/mergeRefs';
import { StatefulPopupProps, StatefulPopupState } from './StatefulPopup.types';
import Popup from '../Popup';

class StatefulPopup extends Component<StatefulPopupProps, StatefulPopupState> {
  private anchorRef = createRef<HTMLElement>();

  public constructor(props: StatefulPopupProps) {
    super(props);
    this.state = {
      visible: false,
    };
  }

  private hide = (): void => {
    this.setState({
      visible: false,
    });
  };

  private toggle = (): void => {
    this.setState((prevState) => ({
      visible: !prevState.visible,
    }));
  };

  private renderAnchor = (): ReactNode => {
    const { children } = this.props;
    if (!children) {
      return null;
    }

    const refKey = typeof children.type === 'string' ? 'ref' : 'innerRef';
    const ref =
      children.props[refKey] == null
        ? this.anchorRef
        : mergeAllRefs(children.props[refKey], this.anchorRef);

    return cloneElement(children, {
      onClick: this.toggle,
      ref,
    });
  };

  private renderPopup = (): ReactNode => {
    const { content, ...restProps } = this.props;
    return (
      <Popup
        {...restProps}
        visible={this.state.visible}
        onOutsideClick={this.hide}
        target="anchor"
        anchor={this.anchorRef}
        children={content}
      />
    );
  };

  public render(): ReactNode {
    return (
      <>
        {this.renderAnchor()}
        {this.renderPopup()}
      </>
    );
  }
}

export default StatefulPopup;
