import * as React from 'react';

interface Props {
  zIndex?: number;
  autoZIndex?: boolean;
  minZIndex?: number;
  anchor?: HTMLElement;
}

interface WrappedComponentProps {
  zIndex?: number;
  anchor?: HTMLElement;
}

const calculateZIndex = (node, minZIndex = 500) => {
  let parent;
  let zIndex = minZIndex;

  if (node) {
    parent = node.parentNode;
  }

  while (parent && parent !== document.body) {
    const style = getComputedStyle(parent);
    const ParentZIndex = style.getPropertyValue('z-index');

    if (!Number.isNaN(Number.parseInt(ParentZIndex, 10))) {
      zIndex = Math.max(zIndex, ParentZIndex as any);
    }

    parent = parent.parentNode;
  }

  return zIndex + 1;
};

export const withAutoZIndex = <TProps extends React.PropsWithChildren<WrappedComponentProps>>(
  WrappedComponent: React.ComponentType<TProps>,
) => {
  return (props: Props & TProps) => {
    let zIndex = props.zIndex || 1000;
    const forwardProps: Props & TProps = { ...props };
    delete forwardProps.autoZIndex;
    if (props.autoZIndex && props.anchor) {
      zIndex = calculateZIndex(props.anchor, zIndex);
    }
    if (props.minZIndex) {
      zIndex = Math.max(props.minZIndex, zIndex);
    }

    return <WrappedComponent {...forwardProps} zIndex={zIndex} />;
  };
};
