import * as React from 'react';
import {
  Select,
  ISelectProps,
  OptionSimple,
  withBaseline,
  withWidthMax,
  Option,
} from '@yandex-lego/components/Select/desktop';
import { withRegistry } from '@bem-react/di';
import { compose } from '@bem-react/core';
import { EMPTY_OPTION_CONTENT } from './Select.constants';
import { selectRegistry } from './selectRegistry';
import { SelectProps, SelectState, SelectValue } from './Select.types';
import { valueToString } from './Select.utils';
import { withMultiple } from './_multiple';
import { withFixKeyDown } from './withFixKeyDown';
import './Select.css';

export const ComposedSelect = compose(
  withBaseline,
  withWidthMax,
  withRegistry(selectRegistry),
  withMultiple,
  withFixKeyDown,
)(Select) as React.FC<ISelectProps>;

class SelectWrap extends React.Component<SelectProps, SelectState> {
  public static defaultProps = {
    view: 'default',
    size: 's',
    hasEmptyValue: true,
    fixKeyDown: true,
    resetItem: '',
  };

  private timer: number;

  public constructor(props: SelectProps) {
    super(props);
    this.state = {
      opened: false,
    };
  }

  public componentDidMount(): void {
    /*
     * Field (react-final-form) некорректно обрабатывает props.input.onChange до своего componentDidMount,
     * поэтому вызываем change с timeout
     * */
    this.timer = window.setTimeout((): void => {
      const { options, value, defaultValueIndex } = this.props;
      if (
        !options ||
        defaultValueIndex == null ||
        defaultValueIndex < 0 ||
        defaultValueIndex > options.length - 1 ||
        (options[defaultValueIndex] as { value?: string }).value == null
      ) {
        return;
      }

      const defaultOption = (options[defaultValueIndex] as OptionSimple).value;
      if (!value && Array.isArray(options)) {
        this.handleChange(defaultOption);
      }
    }, 0);
  }

  public componentWillUnmount(): void {
    window.clearTimeout(this.timer);
  }

  private getOptions = (options: Option[]): Option[] => {
    if (this.props.resetItem && this.props.resetItem !== '') {
      return options.map(valueToString);
    }

    return this.props.hasEmptyValue
      ? [{ value: '', content: EMPTY_OPTION_CONTENT }, ...options.map(valueToString)]
      : options.map(valueToString);
  };

  private parseValue = (value: SelectValue): SelectValue | undefined | null => {
    const { resetItem, value: prevValue } = this.props;

    if (resetItem != null) {
      if (
        prevValue != null &&
        Array.isArray(prevValue) &&
        !prevValue.includes(resetItem) &&
        value.length > 1 &&
        value.includes(resetItem)
      ) {
        const returnValue: string[] = [];
        if (resetItem !== '') {
          returnValue.push(resetItem);
        }
        return returnValue;
      }
      if (Array.isArray(value) && value.length !== 1 && value.includes(resetItem)) {
        return value.filter((v): boolean => v !== resetItem);
      }
    }

    return value;
  };

  private normalizeValue = (
    value: number | number[] | SelectValue | undefined,
  ): SelectValue | undefined => {
    if (value == null) {
      return undefined;
    }
    if (Array.isArray(value)) {
      if (value.length === 1 && value[0] === '') {
        return [];
      }

      return (value as (number | string)[]).map(String);
    }
    return String(value);
  };

  private castTo = (
    value: SelectValue | undefined | null,
  ): string | string[] | number | number[] | undefined | null => {
    const { castTo } = this.props;
    if (castTo && value) {
      if (Array.isArray(value)) {
        return value.map((v) => castTo(v) as number & string);
      }
      return castTo(value);
    }

    return value;
  };

  private handleChange = (event: React.ChangeEvent<HTMLSelectElement> | string): void => {
    const { onChange } = this.props;
    let value: SelectValue;
    if (typeof event === 'string') {
      value = event;
    } else {
      ({ value } = event.target);
    }
    if (onChange) {
      onChange(this.castTo(this.parseValue(value)));
    }
  };

  private handleOpen = (opened: boolean): void => {
    this.setState({ opened });
  };

  public render(): React.ReactNode {
    // redux-form странно обрабатывает onBlur - ставит value = ""
    // похоже что islands select не подписывал onBlur -> не вызывался onBlur в redux-form
    const { onBlur, value, options, ...restProps } = this.props;
    return (
      <ComposedSelect
        {...restProps}
        options={this.getOptions(options)}
        value={this.normalizeValue(value)}
        onChange={this.handleChange}
        opened={this.state.opened}
        setOpened={this.handleOpen}
      />
    );
  }
}

export default SelectWrap as React.ComponentClass<SelectProps, SelectState>;
