import * as React from 'react';
import { Textarea } from '@yandex-lego/components/Textarea/desktop/bundle';
import { mergeAllRefs } from '@yandex-lego/components/lib/mergeRefs';
import throttle from 'lodash/throttle';
import { compose } from '@bem-react/core';
import { getParentOverflows } from './utils/getParentOverflows';
import { withSizeXS } from './_size/Textarea_size_xs';
import { InnerTextareaProps, OuterTextareaProps } from './Textarea.types';
import './Textarea.css';

const ComposedTextarea = compose(withSizeXS)(Textarea) as React.FC<
  Omit<InnerTextareaProps, 'cursorPosition'>
>;

class TextareaWrap extends React.Component<OuterTextareaProps> {
  private static BORDER_WIDTH = 2;

  private static RESIZE_THROTTLE_DELAY = 100;

  private static AUTORESIZE_MAX_HEIGHT = 300;
  private static MAX_HEIGHT_FRACTION = 0.5;

  private textarea: HTMLTextAreaElement | null;

  private initHeight: number;

  private heightByResize: number;

  public static defaultProps = {
    view: 'default',
    size: 's',
    value: undefined,
    cursorPosition: -1,
    rows: '5',
    isEventValue: true,
  };

  private updateMaxHeight = throttle((): void => {
    if (!this.textarea) {
      return;
    }

    const maxHeight = this.props.maxHeight || TextareaWrap.AUTORESIZE_MAX_HEIGHT;

    this.textarea.style.maxHeight = `${Math.min(
      window.innerHeight * TextareaWrap.MAX_HEIGHT_FRACTION,
      maxHeight,
    )}px`;
  }, TextareaWrap.RESIZE_THROTTLE_DELAY);

  private resize = throttle((): void => {
    const { textarea } = this;

    if (!textarea) {
      return;
    }

    const overflows = getParentOverflows(textarea);

    const currentHeight = textarea.offsetHeight;
    if (this.heightByResize !== currentHeight) {
      this.initHeight = currentHeight;
    }

    const maxHeight = this.props.maxHeight || TextareaWrap.AUTORESIZE_MAX_HEIGHT;
    textarea.style.height = 'auto';
    textarea.style.height = `${Math.max(
      this.initHeight,
      Math.min(textarea.scrollHeight + TextareaWrap.BORDER_WIDTH * 2, maxHeight),
    )}px`;

    overflows.forEach((overflow) => {
      overflow.node.scrollTop = overflow.scrollTop;
    });

    this.heightByResize = textarea.offsetHeight;
  }, TextareaWrap.RESIZE_THROTTLE_DELAY);

  public componentDidMount(): void {
    this.setInitHeight();
    this.updateMaxHeight();
    this.moveCursorTo(this.props.cursorPosition);

    window.addEventListener('resize', this.updateMaxHeight);
  }

  public componentDidUpdate(): void {
    this.resize();
  }

  public componentWillUnmount(): void {
    this.resize.cancel();
    this.updateMaxHeight.cancel();
    window.removeEventListener('resize', this.updateMaxHeight);
  }

  private setInitHeight = (): void => {
    if (this.textarea) {
      this.initHeight = this.textarea.offsetHeight;
    }
  };

  private moveCursorTo = (cursorPos: number): void => {
    if (!this.textarea) {
      return;
    }

    let cursorPosition = cursorPos;
    if (cursorPos === -1) {
      cursorPosition = this.textarea.value.length;
    }

    this.textarea.selectionEnd = cursorPosition;
    this.textarea.selectionStart = cursorPosition;
  };

  private getInnerRef = (node: HTMLSpanElement | null): void => {
    const { innerRef, controlRef } = this.props;
    if (node) {
      // https://st.yandex-team.ru/ISL-8754
      this.textarea = node.querySelector('textarea');
    }
    mergeAllRefs(controlRef)(this.textarea);
    mergeAllRefs(innerRef)(node);
  };

  private onChange = (event: React.ChangeEvent<HTMLTextAreaElement>): void => {
    if (typeof this.props.onChange !== 'function') {
      return;
    }

    if (this.props.isEventValue) {
      this.props.onChange(event);
    } else {
      this.props.onChange(event.target.value);
    }
  };

  public render(): React.ReactNode {
    // https://st.yandex-team.ru/CRM-15295
    // eslint-disable-next-line
    const { cursorPosition, isEventValue, componentClassName, ...props } = this.props as any;

    return (
      <ComposedTextarea
        data-testid="Textarea"
        forceNotifyOnBlur={false}
        {...props}
        onChange={this.onChange}
        innerRef={this.getInnerRef}
      />
    );
  }
}

export default TextareaWrap;
