import * as React from 'react';
import { compose } from '@bem-react/core';
import { Textinput } from '@yandex-lego/components/Textinput/desktop/bundle';
import { withSizeXS } from './_size/Textinput_size_xs';
import { withSizeL } from './_size/Textinput_size_l';
import { withWidthS } from './_width/Textinput_width_s';
import { withTypeNumber } from './_type/_number';
import { withHasClear } from './_hasClear';
import { TextinputProps, TextinputWrapProps, TextinputWrapState } from './Textinput.types';

const TextinputWithMod = compose(
  withSizeXS,
  withSizeL,
  withWidthS,
  withTypeNumber,
  withHasClear,
)(Textinput) as React.FC<TextinputProps>;

class TextinputWrap extends React.Component<TextinputWrapProps, TextinputWrapState> {
  private autoFocusTimer: number;

  private autoFocusDelay: number = 0;

  public constructor(props: TextinputWrapProps) {
    super(props);
    this.state = {
      autoFocus: false,
    };
  }

  public componentDidMount(): void {
    // issue: TextInput with autoFocus in Popup
    // mega hack for lego https://st.yandex-team.ru/CRM-7054#1532690789000
    const needAutoFocus = this.needAutoFocus(this.props.autoFocus);
    const delay: number =
      typeof this.props.autoFocus === 'number' ? this.props.autoFocus : this.autoFocusDelay;
    if (needAutoFocus) {
      this.autoFocusTimer = window.setTimeout((): void => {
        if (this) {
          this.setState({ autoFocus: true });
        }
      }, delay);
    }
  }

  public componentWillUnmount(): void {
    window.clearTimeout(this.autoFocusTimer);
  }

  private needAutoFocus = (autoFocusProp: boolean | number | undefined): boolean => {
    return typeof autoFocusProp === 'number' || autoFocusProp || false;
  };

  private handleChange = (event: React.ChangeEvent<HTMLInputElement>): void => {
    const { onChange } = this.props;

    if (onChange) {
      onChange(event.target.value, event);
    }
  };

  public render(): React.ReactNode {
    const autoFocus = this.needAutoFocus(this.props.autoFocus) ? this.state.autoFocus : undefined;
    return (
      <TextinputWithMod
        data-testid="TextInput"
        forceNotifyOnBlur={false}
        {...this.props}
        autoFocus={autoFocus}
        key={`${autoFocus}`}
        onChange={this.handleChange}
      />
    );
  }
}

const ForwardedRefTextinput = React.forwardRef(
  (props: TextinputWrapProps, ref: React.Ref<HTMLSpanElement>) => {
    // https://st.yandex-team.ru/CRM-15295
    const {
      onFocusChange,
      extraProps,
      extraFetchParam,
      minSearchStringLength,
      isDataLoading,
      provider,
      isEventValue,
      hasIcon,
      popupPosition,
      isTouchOnChange,
      ...restProps
      // eslint-disable-next-line
    } = props as any;
    return <TextinputWrap {...restProps} innerRef={props.innerRef || ref} />;
  },
);

ForwardedRefTextinput.defaultProps = {
  size: 's',
  view: 'default',
  hasClear: true,
};

export default ForwardedRefTextinput;
