import { performance } from 'perf_hooks';
import { IncomingMessage, ServerResponse } from 'http';
import webOutgoing from 'http-proxy/lib/http-proxy/passes/web-outgoing';
import { Application, Request, Response } from 'express';
import { indexHTMLProvider } from 'services/IndexHTMLProvider';
import { Config } from 'services/Config';
import { createProxyMiddleware, responseInterceptor } from 'http-proxy-middleware';
import { globalRegistry } from '@yandex-data-ui/monlib-nodejs';
import { CRMBackendErrorCode } from 'typings/CRMBackendErrorCode';
import { secureCRMHeaders, Headers } from 'utils/secureCRMHeaders';

// eslint-disable-next-line @typescript-eslint/no-explicit-any
const webOutgoingArr = Array.from(Object.values(webOutgoing)) as any;

// https://github.com/chimurai/http-proxy-middleware/issues/782
const memoryLeakFix = (proxyRes: IncomingMessage, req: IncomingMessage, res: ServerResponse) => {
    const cleanup = (err?: Error) => {
        // cleanup event listeners to allow clean garbage collection
        proxyRes.removeListener('error', cleanup);
        proxyRes.removeListener('close', cleanup);
        res.removeListener('error', cleanup);
        res.removeListener('close', cleanup);

        // destroy all source streams to propagate the caught event backward
        req.destroy(err);
        proxyRes.destroy(err);
    };

    proxyRes.once('error', cleanup);
    proxyRes.once('close', cleanup);
    res.once('error', cleanup);
    res.once('close', cleanup);
};

export const loadCRMBackendProxy = (app: Application) => {
    const config = Config.getInstance();
    const startProxyWeakMap = new WeakMap<Request, number>();

    const calculateProxyDuration = (req: Request) => {
        const start = startProxyWeakMap.get(req);

        if (!start) {
            return;
        }

        globalRegistry.histogramRate(
            'http.server.backend_proxy.elapsed_time_milliseconds',
            {},
            config.DEFAULT_REQUEST_DURATION_BUCKETS,
        ).record(Math.ceil(performance.now() - start));
    };

    const delegationCleanupSpy = responseInterceptor(async(responseBuffer, proxyRes, req, res) => {
        try {
            const data = JSON.parse(responseBuffer.toString('utf8'));
            if (data.code === CRMBackendErrorCode.DELEGATION_ERROR) {
                (res as Response).clearCookie(config.delegationIdCookieName);
            }
        } catch (error) {
            (req as Request).logger.error(error as Error);
        }

        return responseBuffer;
    });

    app.use('/', createProxyMiddleware({
        target: config.crmBackendApiUrl,
        changeOrigin: true,
        ssl: {
            ca: CA,
        },
        secure: false,
        headers: {
            'X-CRM-Sender': 'frontend',
        },
        selfHandleResponse: true,
        logLevel: 'silent',

        onError(error, req, res) {
            // https://github.com/chimurai/http-proxy-middleware/blob/master/src/_handlers.ts#L56
            // Re-throw error. Not recoverable since req & res are empty.
            if (!req && !res) {
                throw error; // "Error: Must provide a proper URL as target"
            }

            const expressRequest = req as Request;
            const host = req.headers && req.headers.host;
            const code = (error as Error & {code: string}).code ?? 'unknown';

            expressRequest.logger.error(error);
            calculateProxyDuration(expressRequest);
            globalRegistry.rate('http.server.backend_proxy.errors', { code }).inc();

            if (res.writeHead && !res.headersSent) {
                if (/HPE_INVALID/.test(code)) {
                    res.writeHead(502);
                } else {
                    switch (code) {
                        case 'ECONNRESET':
                        case 'ENOTFOUND':
                        case 'ECONNREFUSED':
                        case 'ETIMEDOUT':
                            res.writeHead(504);
                            break;
                        default:
                            res.writeHead(500);
                    }
                }
            }

            res.end(`Error occurred while trying to proxy: ${host}${req.url}`);
        },

        onProxyReq: (proxyReq, req) => {
            const delegationId = (req as Request).cookies[config.delegationIdCookieName];
            if (delegationId) {
                proxyReq.setHeader(config.delegationIdHeaderName, delegationId);
            }

            const devcrm = (req as Request).cookies[config.devLoginCookieName];
            if (devcrm) {
                proxyReq.setHeader(config.devLoginHeaderName, devcrm);
            }

            proxyReq.removeHeader('Cookie');
            proxyReq.setHeader('X-Ya-Service-Ticket', (req as Request).tvm?.crm.tickets?.backend.ticket ?? '');
            proxyReq.setHeader('X-Ya-User-Ticket', (req as Request).blackbox?.userTicket ?? '');

            (req as Request).logger.info({ headers: secureCRMHeaders(proxyReq.getHeaders() as Headers) }, 'CRMBackendProxy: request start');
            startProxyWeakMap.set(req as Request, performance.now());
        },

        onProxyRes: async(proxyRes, req, res) => {
            (req as Request).logger.info({ contentLength: proxyRes.headers['content-length'] }, 'CRMBackendProxy: request end');
            calculateProxyDuration(req as Request);
            res.setHeader('X-Frontend-Version', indexHTMLProvider.version);

            memoryLeakFix(proxyRes, req, res);

            if (proxyRes.statusCode === 403 && proxyRes.headers['content-type'] === 'application/json') {
                await delegationCleanupSpy(proxyRes, req, res);
            } else {
                if (!res.headersSent) {
                    for (let i = 0; i < webOutgoingArr.length; i++) {
                        if (webOutgoingArr[i](req, res, proxyRes, {})) { break }
                    }
                }

                if (!res.finished) {
                    proxyRes.pipe(res);
                }
            }
        },
    }));
};
