import { performance } from 'perf_hooks';
import { Application, Request } from 'express';
import { collectDefaultMetrics, globalRegistry } from '@yandex-data-ui/monlib-nodejs';
import { Config } from 'services/Config';
import { runAfterResponseSended } from 'utils/runAfterResponseSended';

export const DEFAULT_REQUEST_UNKNOWN_PATH = 'unknown';

const getEndpoint = (req: Request) => {
    if (req.route?.endpoint) {
        return req.route.endpoint;
    } else if (req.route?.path.includes('*')) {
        return req.route.path.replace('*', '_');
    }

    if (Array.isArray(req.route?.path)) {
        return req.route.path[0];
    }

    return req.route?.path || DEFAULT_REQUEST_UNKNOWN_PATH;
};

export const loadHTTPMetrics = (app: Application) => {
    collectDefaultMetrics();
    const config = Config.getInstance();

    app.use(function(req, res, next) {
        const start = performance.now();

        globalRegistry.rate('http.server.requests.rps').inc();

        runAfterResponseSended(res, () => {
            const method = req.method;
            const endpoint = getEndpoint(req);

            globalRegistry.rate('http.server.requests.status', { endpoint, method, code: String(res.statusCode) }).inc();

            let metricLabels: Record<string, string> = { endpoint };
            if (endpoint !== DEFAULT_REQUEST_UNKNOWN_PATH) {
                metricLabels.method = method;
            }

            const duration = Math.ceil(performance.now() - start);
            globalRegistry.histogramRate(
                'http.server.requests.elapsed_time_milliseconds',
                metricLabels,
                config.DEFAULT_REQUEST_DURATION_BUCKETS,
            ).record(duration);
        });

        next();
    });
};
