import { Application } from 'express';
import expressBlackbox, { ExpressBlackboxOptions } from '@yandex-int/express-blackbox';
import expressTvm from '@yandex-int/express-tvm';
import cookieParser from 'cookie-parser';
import { passportRedirect } from 'middlewares/passportRedirect';
import { addUserLoginFromRaw } from 'middlewares/addUserLoginFromRaw';
import { devUserIpMiddleware } from 'middlewares/devUserIpMiddleware';
import { secureTVMError } from 'middlewares/secureTVMError';
import { Config } from 'services/Config';
import { getLocalDevYandexRealIp } from 'utils/getLocalDevYandexRealIp';

export const loadYandexAuth = (app: Application) => {
    const config = Config.getInstance();

    const expressBlackboxOptions: ExpressBlackboxOptions = {
        api: 'blackbox.yandex-team.ru',
        attributes: {},
        getServiceTicket: req => {
            if (
                req.tvm &&
                req.tvm.crm &&
                req.tvm.crm.tickets &&
                req.tvm.crm.tickets.blackbox
            ) {
                return req.tvm.crm.tickets.blackbox.ticket;
            }
        },
    };

    if (config.isLocalEnv) {
        expressBlackboxOptions.getYaboxOptions = req => ({
            userip: config.devUserIp || req.devUserIp || getLocalDevYandexRealIp(),
        });
    }

    app.use(cookieParser());
    app.use((req, _, next) => {
        req.logger.info('tvm request start');
        next();
    });
    app.use(expressTvm({
        clientId: 'crm',
        destinations: ['blackbox', 'backend'],
        serverUrl: config.tvmApiUrl,
        token: config.tvmApiToken,
        throwError: true,
        logError: false,
        timeout: 500,
        retry: 2,
    }));
    app.use(secureTVMError);

    app.use((req, _, next) => {
        req.logger.info('tvm request end and blackbox request start');
        next();
    });

    if (config.isLocalEnv && config.useDevUserIpUrl) {
        app.use(devUserIpMiddleware({ url: config.devUserIpUrl }));
    }

    app.use(expressBlackbox(expressBlackboxOptions));
    app.use(addUserLoginFromRaw);
    app.use((req, _, next) => {
        req.logger.info('blackbox request end');
        next();
    });
    app.use(passportRedirect());
};
