import { Request, Response } from 'express';
import { globalRegistry } from '@yandex-data-ui/monlib-nodejs';
import got, { Got, Headers } from 'got';
import { Config } from 'services/Config';
import { CRMBackendErrorCode } from 'typings/CRMBackendErrorCode';
import { GotLogger } from 'services/GotLogger';

export interface CRMBackendClientOptions {
    request: Request;
    response: Response;
}

export class CRMBackendClient {
    static create(options: CRMBackendClientOptions) {
        return new CRMBackendClient(options).client;
    }

    public readonly client: Got;

    private config: Config = Config.getInstance();

    constructor(private options: CRMBackendClientOptions) {
        this.client = this.createClient();
    }

    private createClient() {
        const { request: req, response: res } = this.options;
        const { config } = this;

        const gotLogger = new GotLogger('CRMBackendClient', req.logger);

        return got.extend({
            prefixUrl: config.crmBackendApiUrl,
            responseType: 'json',
            headers: this.createDefaultHeaders(),
            https: { certificateAuthority: CA },
            hooks: {
                beforeRequest: [
                    options => {
                        gotLogger.logRequest(options);
                    },
                ],
                afterResponse: [
                    response => {
                        gotLogger.logResponse(response);
                        return response;
                    },
                ],
                beforeError: [
                    error => {
                        const { response } = error;
                        globalRegistry.rate('services.crm_backend.error', { code: error.code }).inc();

                        gotLogger.logError(error);

                        if (response && response.body) {
                            error.name = 'CRMBackendRequestError';
                            // eslint-disable-next-line @typescript-eslint/no-explicit-any
                            error.message = `${(response.body as any).message ?? 'no message'} (${response.statusCode})`;

                            if ((response.body as { code: string }).code === CRMBackendErrorCode.DELEGATION_ERROR) {
                                res.clearCookie(this.config.delegationIdCookieName);
                                error.code = CRMBackendErrorCode.DELEGATION_ERROR;
                            }
                        }

                        return error;
                    },
                ],
            },
        });
    }

    private createDefaultHeaders(): Headers {
        const { request: req } = this.options;

        const headers: Headers = {};

        const delegationId = req.cookies[this.config.delegationIdCookieName];
        if (delegationId) {
            headers[this.config.delegationIdHeaderName] = delegationId;
        }

        const devcrm = req.cookies[this.config.devLoginCookieName];
        if (devcrm) {
            headers[this.config.devLoginHeaderName] = devcrm;
        }

        headers['X-Ya-Service-Ticket'] = req.tvm?.crm.tickets?.backend.ticket ?? '';
        headers['X-Ya-User-Ticket'] = req.blackbox?.userTicket ?? '';

        return headers;
    }
}
