import { exponentialBuckets } from '@yandex-data-ui/monlib-nodejs';
import { LoggerMode } from './Config.types';

export class Config {
    public static LoggerMode = LoggerMode;

    private static instance: Config | null = null;

    static getInstance() {
        if (Config.instance) {
            return Config.instance;
        }

        return Config.instance = new Config();
    }

    readonly pathToStatic: string;
    readonly tvmApiUrl: string;
    readonly bunkerApiUrl: string;
    readonly tvmApiToken: string;
    readonly crmBackendApiUrl: string;
    readonly crmAppUrl: string;
    readonly loggerMode: LoggerMode;

    readonly appKey: string;

    readonly kibanaLogPath: string = '';

    readonly delegationIdCookieName = 'delegationId';
    readonly delegationIdHeaderName = 'X-Crm-Delegation-Id';

    readonly devLoginCookieName = 'devcrm';
    readonly devLoginHeaderName = 'X-Crm-Dev-Login';

    readonly isLocalEnv: boolean = false;

    readonly devUserIp: string = '';

    readonly useDevUserIpUrl: boolean = false;
    readonly devUserIpUrl: string;

    /**
     * По-умолчанию для реквестов создаем бакеты от 16мс до 131072мс
     * Все что выполнялось дольше будет записано в бакет +inf
     */
    readonly DEFAULT_REQUEST_DURATION_BUCKETS = exponentialBuckets(16, 2, 14);

    constructor() {
        let mdsEndpoint = process.env.MDS_ENDPOINT ?? '';
        this.pathToStatic = process.env.MDS_STATIC_BUILD_PATH ?? '';

        this.bunkerApiUrl = process.env.BUNKER_API_URL ?? '';

        this.tvmApiUrl = process.env.DEPLOY_TVM_TOOL_URL ?? '';
        this.tvmApiToken = process.env.TVMTOOL_LOCAL_AUTHTOKEN ?? '';

        this.crmBackendApiUrl = process.env.CRM_BACKEND_API_URL ?? '';
        this.crmAppUrl = process.env.CRM_APP_URL ?? '';

        this.loggerMode = (process.env.LOGGER_MODE as LoggerMode) ?? LoggerMode.Line;

        this.appKey = process.env.APP_KEY ?? '';

        this.kibanaLogPath = process.env.KIBANA__LOG_PATH ?? '';

        this.isLocalEnv = Boolean(process.env.IS_LOCAL_ENV);

        this.devUserIp = process.env.DEV_USER_IP ?? '';

        this.useDevUserIpUrl = Boolean(process.env.USE_DEV_USER_IP_URL);
        this.devUserIpUrl = process.env.DEV_USER_IP_URL ?? 'https://st.yandex-team.ru/ping/ip';

        if (!this.pathToStatic) {
            throw new Error('MDS_STATIC_BUILD_PATH is required.');
        }

        if (this.pathToStatic.startsWith('s3://')) {
            if (!mdsEndpoint) {
                throw new Error('MDS_ENDPOINT is required if MDS_STATIC_BUILD_PATH starts with s3://.');
            }

            this.pathToStatic = mdsEndpoint + this.pathToStatic.slice(4);
        }

        if (!this.tvmApiUrl) {
            throw new Error('DEPLOY_TVM_TOOL_URL is required.');
        }

        if (!this.tvmApiToken) {
            throw new Error('TVMTOOL_LOCAL_AUTHTOKEN is required.');
        }

        if (!this.crmBackendApiUrl) {
            throw new Error('CRM_BACKEND_API_URL is required.');
        }

        if (!this.crmAppUrl) {
            throw new Error('CRM_APP_URL is required.');
        }

        if (!this.bunkerApiUrl) {
            throw new Error('BUNKER_API_URL is required.');
        }

        if (!this.appKey) {
            throw new Error('APP_KEY is required.');
        }

        if (!Object.values(LoggerMode).includes(this.loggerMode as LoggerMode)) {
            throw new Error(`LOGGER_MODE support only: ${Object.values(LoggerMode)}.`);
        }
    }
}
