import logging
import random

from aiohttp import web

from crm.infra.github_webhook.pr_events.base import signed_event
from crm.infra.github_webhook.utils.bb import PullRequestInfo
from crm.infra.github_webhook.utils.slack import post_message
from crm.infra.github_webhook.utils.available_developers_syncer import get_available_developers

logger = logging.getLogger(__name__)

# ревьюверы выбираются с минимальным скорингом
SCORING_FOR_PR_DIFF_LIST = lambda pr_diff_list: sum(pr_diff[0] + pr_diff[1] for pr_diff in pr_diff_list)


@signed_event
async def process_opened_pr(request):
    logger.info('processing opened pr')
    body = await request.json()
    info = PullRequestInfo(body['pullRequest'])

    message = f':mag: {info.author_name} открыл пул-реквест и добавил тебя ревьюером\n'+ await info.get_pr_summary()

    for reviewer in info.reviewers:
        await post_message(reviewer, message)

    max_reviewers_count = 2
    if any(line.strip() == '/noreview' for line in info.description.splitlines()):
        max_reviewers_count = 0
    elif any(line.strip() == '/singlereview' for line in info.description.splitlines()):
        max_reviewers_count = 1

    reviewers_to_add_count = max(0, max_reviewers_count - len(info.reviewers))
    logger.info(f'reviewers_to_add_count = {reviewers_to_add_count}')

    potential_reviewers = await get_available_developers()
    scored_potential_reviewers = [(login, SCORING_FOR_PR_DIFF_LIST(data['prs_diffs'])) for login, data in potential_reviewers]
    # шафлим, чтобы при одинаковом скоринге выбирался случайный ревьювер
    random.shuffle(scored_potential_reviewers)
    scored_potential_reviewers.sort(key=lambda item: item[1])
    logger.info(f'scored_potential_reviewers = {scored_potential_reviewers}')

    picked_reviewers = [login for login, _ in scored_potential_reviewers[:reviewers_to_add_count]]
    logger.info(f'picked_reviewers = {picked_reviewers}')
    info.add_reviewers(picked_reviewers)

    return web.Response(status=200)


@signed_event
async def process_changed_reviewers(request):
    logger.info('processing changed reviewers')
    body = await request.json()
    info = PullRequestInfo(body['pullRequest'])

    message_added = f':mag: {info.author_name} добавил тебя ревьюером в пул-реквест\n'+ await info.get_pr_summary()
    message_removed = f':neutral_face: {info.author_name} удалил тебя из ревьюеров пул-реквеста\n'+ await info.get_pr_summary()

    for reviewer in info.added_reviewers:
        await post_message(reviewer, message_added)

    for reviewer in info.removed_reviewers:
        await post_message(reviewer, message_removed)

    return web.Response(status=200)


@signed_event
async def process_pr_approved(request):
    logger.info('processing approved pr')
    body = await request.json()
    info = PullRequestInfo(body['pullRequest'])

    if info.approved_by:
        message = f':+1: {info.approved_by} заапрувил пул-реквест\n'+ await info.get_pr_summary()
        await post_message(info.author, message)

    if info.all_approved:
        message = ':+1: Все ревьюеры заапрувили пул-реквест\n'+ await info.get_pr_summary()
        await post_message(info.author, message)

    return web.Response(status=200)


@signed_event
async def process_pr_unapproved(request):
    logger.info('processing unapproved pr')
    body = await request.json()
    info = PullRequestInfo(body['pullRequest'])

    if info.unapproved_by:
        message = f':neutral_face: {info.unapproved_by} снял апрув/нидворк с пул-реквеста\n'+ await info.get_pr_summary()
        await post_message(info.author, message)

    return web.Response(status=200)


@signed_event
async def process_pr_needs_work(request):
    logger.info('processing need work')
    body = await request.json()
    info = PullRequestInfo(body['pullRequest'])

    if info.needs_work_by:
        message = f':pencil: {info.needs_work_by} запросил доработок по пул-реквесту\n'+ await info.get_pr_summary()
        await post_message(info.author, message)

    return web.Response(status=200)
