import logging
import aiohttp
import asyncio
import os

from time import time

from crm.infra.github_webhook.utils.bb import get_all_prs

logger = logging.getLogger(__name__)

DEVELOPER_ABC_ROLE_CODE = 'developer'
ABC_SERVICE_SLUG = os.getenv('ABC_SERVICE_SLUG')

ABC_API_TOKEN = os.getenv('ABC_API_TOKEN')
STAFF_API_TOKEN = os.getenv('STAFF_API_TOKEN')

SYNC_AVAILABLE_DEVELOPERS_LOCK = asyncio.Lock()
SYNC_AVAILABLE_DEVELOPERS_INTERVAL = 10 * 60  # 10 min

AVAILABLE_DEVELOPERS = {
    'last_sync': None,
    'devs': {}
}


async def get_available_developers():
    async with SYNC_AVAILABLE_DEVELOPERS_LOCK:
        last_sync = AVAILABLE_DEVELOPERS['last_sync']
        if last_sync and time() - last_sync < SYNC_AVAILABLE_DEVELOPERS_INTERVAL:
            logger.info('skip syncing available developers')
            return

        logger.info('a0')
        uri = 'https://abc-back.yandex-team.ru/api/v4/services/members'
        params = {'role__code': DEVELOPER_ABC_ROLE_CODE, 'service__slug': ABC_SERVICE_SLUG}
        logger.info(uri)
        logger.info(params)
        logger.info({'Authorization': f'OAuth {ABC_API_TOKEN}'})
        async with aiohttp.ClientSession(headers={'Authorization': f'OAuth {ABC_API_TOKEN}'}) as session:
            logger.info('a1')
            try:
                async with session.get(uri, params=params) as resp:
                    logger.info('a2')
                    json = await resp.json()
                    logger.info('a3')
                    AVAILABLE_DEVELOPERS['devs'] = {result['person']['login']: {} for result in json['results']}
                    logger.info('a4')
                    logger.info(f'Loaded {len(AVAILABLE_DEVELOPERS["devs"])} developers from ABC')
            except Exception as e:
                logger.error(type(e))
                logger.error(e)
                raise
        logger.info('a5')

        uri = 'https://staff.yandex-team.ru/gap-api/api/export_gaps'
        params = [('l', dev_login) for dev_login in AVAILABLE_DEVELOPERS["devs"]] + [('field', 'work_in_absence')]
        async with aiohttp.ClientSession(headers={'Authorization': f'OAuth {STAFF_API_TOKEN}'}) as session:
            async with session.get(uri, params=params) as resp:
                json = await resp.json()
                for login, data in json['persons'].items():
                    if any(not gap['work_in_absence'] for gap in data):
                        logger.info(f'{login} is absent')
                        del AVAILABLE_DEVELOPERS['devs'][login]
        logger.info(f'Left with {len(AVAILABLE_DEVELOPERS["devs"])} developers after filtering out absent ones')

        prs = await get_all_prs()
        prs_with_diffs = [{'pr': pr, 'diff': await pr.get_diff_summary()} for pr in prs]
        for login, data in AVAILABLE_DEVELOPERS['devs'].items():
            data['prs_diffs'] = [pr_with_diff['diff'] for pr_with_diff in prs_with_diffs if login in pr_with_diff['pr'].reviewers]
            logger.info(f'{login} has {len(data["prs_diffs"])} pull requests to review')

        AVAILABLE_DEVELOPERS['last_sync'] = time()

    return AVAILABLE_DEVELOPERS['devs']
