import logging
import os

import aiohttp

from crm.infra.github_webhook.utils.git import get_diff_summary

logger = logging.getLogger(__name__)
REPO = os.getenv('REPO')
PROJECT = os.getenv('PROJECT')
BB_API_TOKEN = os.getenv('BB_API_TOKEN')


class PullRequestInfo:
    def __init__(self, json) -> None:
        self.author_name = json['author']['user']['displayName']
        self.author = json['author']['user']['slug']

        self.description = json.get('description', '')

        reviewers_json = json['reviewers']
        self.reviewers = [r['user']['slug'] for r in reviewers_json]

        self.version = json['version']

        self.id = json['id']
        link = f'https://bb.yandex-team.ru/projects/{PROJECT}/repos/{REPO}/pull-requests/{self.id}'
        title = json['title']

        self.link = f'<{link}|{title}>'

        added_reviewers_json = json.get('addedReviewers', [])
        removed_reviewers_json = json.get('removedReviewers', [])

        self.added_reviewers = [r['slug'] for r in added_reviewers_json]
        self.removed_reviewers = [r['slug'] for r in removed_reviewers_json]

        self.approved_by = None
        self.unapproved_by = None
        self.needs_work_by = None
        participant = json.get('participant')
        if participant:
            user = participant['user']['displayName']
            status = participant['status']
            if status == 'APPROVED':
                self.approved_by = user
            elif status == 'UNAPPROVED':
                self.unapproved_by = user
            elif status == 'NEEDS_WORK':
                self.needs_work_by = user

        self.all_approved = all([reviewer['status'] == 'APPROVED' for reviewer in reviewers_json])
        self.not_approved_yet = [r['user']['slug'] for r in reviewers_json if r['status'] != 'APPROVED']

    async def add_reviewers(self, reviewers_names):
        uri = f'https://bb.yandex-team.ru/rest/api/1.0/projects/{PROJECT}/repos/{REPO}/pull-requests/{self.id}'
        body = {'version': self.version, 'reviewers': [{'user': {'name': reviewer_name}} for reviewer_name in reviewers_names]}
        async with aiohttp.ClientSession(headers={'Authorization': f'OAuth {BB_API_TOKEN}'}) as session:
            async with session.put(uri, body=body) as resp:
                if resp.ok:
                    self.__init__(await resp.json())

    async def get_diff_summary(self):
        uri = f'https://bb.yandex-team.ru/rest/api/1.0/projects/{PROJECT}/repos/{REPO}/pull-requests/{self.id}.diff'
        async with aiohttp.ClientSession(headers={'Authorization': f'OAuth {BB_API_TOKEN}'}) as session:
            async with session.get(uri) as resp:
                diff = await resp.text()
                return get_diff_summary(diff)

    async def get_pr_summary(self):
        pluses, minuses = await self.get_diff_summary()
        return f'+{pluses} -{minuses} {self.link}'


async def get_all_prs():
    pull_requests = []
    is_last_page = False
    start = 0
    while not is_last_page:
        uri = f'https://bb.yandex-team.ru/rest/api/1.0/projects/{PROJECT}/repos/{REPO}/pull-requests'
        params = {'start': start}
        async with aiohttp.ClientSession(headers={'Authorization': f'OAuth {BB_API_TOKEN}'}) as session:
            async with session.get(uri, params=params) as resp:
                logging.debug('Get pull requests')
                json = await resp.json()

                pull_requests.extend(PullRequestInfo(pr) for pr in json['values'])

                is_last_page = json['isLastPage']
                start = json.get('nextPageStart')
                if not is_last_page:
                    logging.debug('Get more pull requests')

    logging.debug(f'Got {len(pull_requests)} pull requests')

    return pull_requests
