import logging
import hashlib
import hmac
import os


logger = logging.getLogger(__name__)


async def check_sha256_signature(request):
    return await __check_signature(request, 'sha256', hashlib.sha256)


async def check_sha1_signature(request):
    return await __check_signature(request, 'sha1', hashlib.sha1)


async def __check_signature(request, name, hashfunc):
    token = os.environ['GITHUB_TOKEN'].encode('utf-8')
    body = await request.read()

    header_signature = request.headers['X-Hub-Signature']
    if not header_signature:
        logger.error('X-Hub-Signature header not found in request')
        return False

    actual_signature = name + '=' + hmac.new(token, body, hashfunc).hexdigest()

    result = hmac.compare_digest(header_signature, actual_signature)
    if not result:
        logger.error('X-Hub-Signature value is invalid')

    return result
